/*****************************************************************************
* 
*  This file is part of GPIO232-16.
*  
*  GPIO232-16 is free software: you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation, either version 3 of the License, or
*  (at your option) any later version.
*  
*  GPIO232-16 is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*  
*  You should have received a copy of the GNU General Public License
*  along with GPIO232-16.  If not, see <http://www.gnu.org/licenses/>.
*
 ****************************************************************************/
 
#include <project.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <cypins.h>

#include "main.h"
#include "analog.h"
#include "digital.h"
#include "interrupt.h"
#include "io.h"

////////////////////////////////////
int    g_echo;
int    g_boolfunc_byte;
int    g_baud_rate;
int    g_hex_case;
int    g_monitor_mode;
int    g_monitor_ADC;
int    g_adc_channels;
int    g_timestamp;
int    g_output_base;
uint16 g_in_out_mask;
uint16 g_out_mode_mask;
uint16 g_pull_up_mask;
uint16 g_trig_rising;
uint16 g_trig_falling;

// LED mode switches between a 5k pull up
// and a 5k pull down resistor
// can be used as input for switches
uint16 g_led_mode_mask;


// EEPROM storage in flash, this is declared in conf_rom.h
const uint8_t conf_rom_em_EepromStorage[conf_rom_ACTUAL_SIZE] __ALIGNED(CY_FLASH_SIZEOF_ROW) = {0u};


////////////////////////////////////
int main()
{
	init_gpio();
	user_led_Write(1); // init marker
	conf_rom_Init((uint32_t)conf_rom_em_EepromStorage);
	
	UART_1_Start();
	RTC_Start();
	IDAC_1_Start();
	IDAC_2_Start();
	ADC_1_Start();
	ADC_1_StartConvert();
	clock_1kHz_Start();
	
    CyGlobalIntEnable;
	
	ty_config conf;
	load_config(&conf);

	// init ////////////
	g_echo            = conf.echo;
	g_hex_case        = conf.hex_case == 0 ? 0 : 16; // default lower case
	line_break_func   = get_line_break_func(conf.line_break); // line_break_CRLF
	isr_edge_out_func = get_isr_edge_func(conf.edge_trig); // isr_edge_out_word
	g_boolfunc_byte   = LOW_BYTE;
	g_monitor_mode    = 0;
	g_monitor_ADC     = 0;
	g_adc_channels    = conf.adc_channel;	// ADC_CHANNEL_1
	g_timestamp       = conf.timestamp;  	// no timestamp
	g_output_base     = conf.read_base;
	
	g_trig_rising     = 0x0000;
	g_trig_falling    = 0x0000;
	set_edge_trigger();

	set_baud_rate(conf.BAUD_rate);
	clock_pGen_SetDividerValue(conf.pgen_clock == 0 ? 48u : 48000u);
	set_trig_readout(0, 3); // ext trig
	set_ext_trig_edge(EGDGE_TRIG_DISABLE);
	int_ext_trig_StartEx(isr_ext_trig);
	
	if(conf.opamp == 1) opAmp_dac2_Start();

	user_led_Write(0);
	UART_1_UartPutChar('p');
	
	int  led = 0;
	int  bt  = 0; // byte
	int  wd  = 0; // word
	char ch  = 0; // character from serial input
	int  dec_input = -1;
	
	user_led_Write(1); // ON signal
	
	for(;;){
		bt = wd = 0;
		switch(ch = UART_getc()){
			case '#': // convert dec -> hex
				convert();
				break;
				
			case 'P': // enable pulse generator
				wd = (dec_input < 0 ? getword() : dec_input);
				if(wd < 0) break;
				set_pGen(wd, pGen_ReadCompare());
				break;
				
			case 'L': // set pulse width
				wd = (dec_input < 0 ? getword() : dec_input);
				if(wd < 0) break;
				set_pGen(pGen_ReadPeriod(), wd);
				break;
				
			case '\r':
				// enables line break on [Enter] key,
				// for more convenience
				if(g_echo == 0) UART_1_UartPutChar('\r');
				UART_1_UartPutChar('\n');
				break;
				
			case 'l': // set low byte
				bt = (dec_input < 0 ? getbyte() : dec_input);
				if(bt < 0) break;
				set_low_byte(bt);
				break;
				
			case 'h': // set high byte
				bt = (dec_input < 0 ? getbyte() : dec_input);
				if(bt < 0) break;
				set_high_byte(bt);
				break;
				
			case ' ':
			case 'q':
				show_status(ch == 'q' ? 1 : 0);
				break;
			
			case 'w': // set word
				wd = (dec_input < 0 ? getword() : dec_input);
				if(wd < 0) break;
				set_word(wd);
				break;
				
			case 's': // set bit
			case 'z': // zero bit
				bt = set_reset_bit(ch, dec_input);
				break;

			case 'a': // port = port & <byte>
			case 'o': // port = port | <byte>
			case 'x': // port = port ^ <byte>
				bt = boolfunc_byte(ch, dec_input);
				break;
				
			case 'A': // port = port & <byte>
			case 'O': // port = port | <byte>
			case 'X': // port = port ^ <byte>
				bt = boolfunc_word(ch, dec_input);
				break;
				
			case 'i': // invert low byte
				set_low_byte(~LOBYTE(get_data_cache()));
				break;

			case 'I': // invert high byte
				set_high_byte(~HIBYTE(get_data_cache()));
				break;
				
			case 'N': // invert word (negate)
				set_word(~get_data_cache());
				break;
				
			case 'r': // read low byte
				send_timestamp();
				byte_out(IO_low_Read());
				line_break_func();
				break;
				
			case 'R': // read high byte
				send_timestamp();
				byte_out(IO_high_Read());
				line_break_func();
				break;
				
			case 'D': // read word
				send_timestamp();
				word_out(MAKEWORD(IO_low_Read(), IO_high_Read()));
				line_break_func();
				break;
				
			case 'c': // clear low byte
				set_low_byte(0);
				break;
			
			case 'C': // clear high byte
				set_high_byte(0);
				break;
				
			case 'F': // clear word
				set_word(0);
				break;
				
			case 'M': // start/stop monitor
				init_monitor_mode();
				break;
				
			case '*': // enter config command
				bt = config();
				break;
				
			case '.': // cancel input
				bt = -1;
				break;
				
			case '?': // show useage as single page
				usage(0);
				break;
				
			case 'S': { // read ADC
				int16 res;
				send_timestamp();
				if((g_adc_channels + 1) & 1){
					res = ADC_1_GetResult16(0);
					word_out(res);
				}
				if(g_adc_channels == ADC_CHANNEL_BOTH) UART_1_UartPutChar(':');
				
				if((g_adc_channels + 1) & 2){
					res = ADC_1_GetResult16(1);
					word_out(res);
				}
				line_break_func();
				break;
			}
				
			case '~': // enter analog command
				bt = analog(dec_input);
				break;
			
			case 27: { // ESC
				init_readout(0, NULL);
				int_edge_low_Stop();
				int_edge_high_Stop();
				if(g_monitor_mode == 1) init_monitor_mode();
				if(g_monitor_ADC == 1) init_adc_monitor();
				UART_1_UartPutChar('!');
				break;
			}
				
			case 'b': // send burst of bytes
			case 'B':
				bt = GETBYTE_BREAK_IF_NEG;
				bt = send_burst_byte(ch, bt);
				break;
				
			case 'W': // send burst of words
				bt = GETBYTE_BREAK_IF_NEG;
				bt = send_burst_word(bt);
				break;
		}
		
		// if a continuous sequence of number digits is entered
		// their value is collected as a decimal value
		// any other character will reset the collection of digits
		if(ch >= '0' && ch <= '9'){
			// interpret as decimal input
			if(dec_input < 0) dec_input = 0; // init @ 1st digit
			dec_input = (dec_input * 10) + (ch - '0');
		}else{
			// reset decimal input
			dec_input = -1;
		}
		
		// acknowledge transmisstion break (indicated by receiving a '.')
		if(bt < 0 || wd < 0){
			UART_1_UartPutChar('#');
			dec_input = -1; // just in case
		}

		user_led_Write(led);
		led = 1 - led;
	}
	
	return 0; // never reached
}


////////////////////////////////////
int config()
{
	int wd = 0;
	int bt = 0;
	int ack = 0;
	
	switch(UART_getc()){
		case 'B': {
			char ch = UART_getc();
			if(ch == '.') return -1;
			
			ack = 1;
			if(ch == '0')
				g_boolfunc_byte = LOW_BYTE;
			else
				g_boolfunc_byte = HIGH_BYTE;
			
			break;
		}
		
		case '?': // show usage in multiple pages with 24 or 40 lines
			usage(1);
			break;
		
		case 'D': // set pin direction (1=in, 0=out; default FFFF)
			wd = GETWORD_RETURN_IF_NEG;
			g_in_out_mask = wd;
			set_drive_mode();
			ack = 1;
			break;
			
		case 'O': // set output type
			wd = GETWORD_RETURN_IF_NEG;
			g_out_mode_mask = wd;
			set_drive_mode();
			ack = 1;
			break;
			
		case 'P': // enable strong pull-up
			wd = GETWORD_RETURN_IF_NEG;
			g_pull_up_mask = wd;
			set_drive_mode();
			ack = 1;
			break;
			
		case 'L': // LED mode
			wd = GETWORD_RETURN_IF_NEG;
			g_led_mode_mask = wd;
			set_drive_mode();
			ack = 1;
			break;
			
		case 'p': // ping
			UART_1_UartPutString("!PONG");
			break;
			
		case 'V': // get FW version; answer: "!<Version>"
			UART_1_UartPutString("!" GPIO232_16_VERSION);
			break;
			
		case 'Y':
			bt = getbyte();
			switch(bt){
				case 0x42: // reset
					CySoftwareReset();
					return -1;
					
				case 0x99: { // restore factory defaults
					ty_config_fields conf;
					load_defaults(&conf.fields);
					conf_rom_Write(0, conf.conf_array, CONF_BLOCK_SIZE);
					ack = 1;
					break;
				}
			}
			break;
			
		case 'X': // system function
			bt = GETBYTE_RETURN_IF_NEG;
			
			switch(bt & 0xF0){
				case 0x00: // echo
					g_echo = bt & 0x0F;
					break;
				
				case 0x10: // line break function
					line_break_func = get_line_break_func(bt & 0x0F);
					break;
					
				case 0x20: // set baud rate
					set_baud_rate(bt & 0x0F);
					break;
					
				case 0x30: // case of hex letters
					if((bt & 0x0F) == 0)
						g_hex_case = 0;
					else
						g_hex_case = 16; // shift to second set of characters
					
					break;
					
				case 0x40: // set ADC readout channel
					g_adc_channels = bt & 0x03;
					if(g_adc_channels > 2) g_adc_channels = 2;
					break;
					
				case 0x50: // triggered readout data source
					isr_edge_out_func = get_isr_edge_func(bt & 0x0F);
					break;
					
				case 0x60: // set pGen clock frequency
					if((bt & 0x0F) == 0)
						clock_pGen_SetDividerValue(48u); // 1MHz
					else
						clock_pGen_SetDividerValue(48000u); // 1kHz

					break;
						
				case 0x70: // set timestamp type
					g_timestamp = bt & 0x03;
					if(g_timestamp > 2) g_timestamp = 2;
					break;
					
				case 0x80: // default readout base: x: 0=HEX, 1=DEC
					g_output_base = ( (bt & 0x0F) == 0 ? HEX : DEC );
					break;
					
				case 0x90: // enable opamp
					if((bt & 0x0F) == 0)
						opAmp_dac2_Stop();
					else
						opAmp_dac2_Start();
						
					break;
			}
			
			save_config();
			
			ack = 1;
			break;
			
		case 'r': // read low byte repeatedly every <nnnn> msec
			wd = GETWORD_RETURN_IF_NEG;
			init_readout(wd, isr_read_timer_low);
			ack = 1;
			break;
			
		case 'R': // read high byte repeatedly
			wd = GETWORD_RETURN_IF_NEG;
			init_readout(wd, isr_read_timer_high);
			ack = 1;
			break;
			
		case 'W': // read word repeatedly
			wd = GETWORD_RETURN_IF_NEG;
			init_readout(wd, isr_read_timer_word);
			ack = 1;
			break;
			
		case 't': // readout on falling edge
			wd = GETWORD_RETURN_IF_NEG;
			g_trig_falling = wd;
			set_edge_trigger();
			ack = 1;
			break;
			
		case 'T': // read high byte on change
			wd = GETWORD_RETURN_IF_NEG;
			g_trig_rising = wd;
			set_edge_trigger();
			ack = 1;
			break;
			
		case 'C': // show config string
			send_config();
			break;
			
		case 'E': // config external triggering
			wd = GETWORD_RETURN_IF_NEG;
			set_ext_trig(wd);
			ack = 1;
			break;
			
		case 'Z': // set time (NYI)
			bt = set_time();
			if(bt < 0) return -1;
			ack = 1;
			break;
			
		case '.': // cancel input
			return -1;
	}
	
	if(ack) UART_1_UartPutChar('!');
	
	return 0;
}

////////////////////////////////////
int set_reset_bit(char ch, int dec_inp)
{
	int bit = 0;
	
	if(dec_inp < 0){
		bit = UART_getc();
		if(bit == '.') return -1;
		if(bit < '0' || bit > 'f') return 0;
		if(bit > '9' && bit < 'a') return 0;
		if(bit > '9')
			bit -= 'a' - 10;
		else
			bit -= '0';
	}else{
		bit = dec_inp & 0x0F;
	}
	
	set_word(
		(get_data_cache() & ~(1 << bit)) // always clear bit
		|
		((ch == 's' ? 1 : 0) << bit) // set/don't set, depending on char
	);
	
	return 0;
}

////////////////////////////////////
int set_time()
{
	return 0;
}

////////////////////////////////////
void convert()
{
	UART_1_UartPutString("enter dec (0..65535): ");
	
	char ch;
	long val = 0;
	
	while((ch = UART_getc_ne()) != '.'){
		if(ch == '\r') break;
		if(ch == 27) break; // ESC
		if(ch == 127){ // backspace
			val /= 10;
			UART_1_UartPutChar(ch); // echo
			continue;
		}
		// let through '0' .. '9', only
		if(ch < '0') continue;
		if(ch > '9') continue;
		
		UART_1_UartPutChar(ch); // echo
		val = (val*10) + (ch - '0');
	}
	
	if(ch == 27) return; // literally "escape"
	
	line_break_CRLF();
	
	put_hex(HIBYTE(val));
	put_hex(LOBYTE(val));
	
	line_break_CRLF();
}

////////////////////////////////////
void save_config()
{
	ty_config_fields conf;
	get_config(&(conf.fields));
	
	// make sure buildmarker is always set
	conf.fields.new_build_marker = CONF_BUILDMARKER;
	
	conf_rom_Write(0, conf.conf_array, CONF_BLOCK_SIZE);
}

////////////////////////////////////
void load_config(ty_config* conf)
{
	ty_config_fields conf_buf = { 0 };
	_EERET ret = conf_rom_Read(0, conf_buf.conf_array, CONF_BLOCK_SIZE);
	
	if(ret != CY_EM_EEPROM_SUCCESS || conf_buf.fields.new_build_marker != CONF_BUILDMARKER){
		load_defaults(&conf_buf.fields);
		conf_rom_Write(0, conf_buf.conf_array, CONF_BLOCK_SIZE);
	}
	
	*conf = conf_buf.fields;
}

////////////////////////////////////
void load_defaults(ty_config* conf)
{
	conf->echo        = 0;
	conf->line_break  = 3; 				// '\r\n'
	conf->edge_trig   = 0; 				// word
	conf->adc_channel = ADC_CHANNEL_1;
	conf->timestamp   = 0; 				// no timestamp
	conf->BAUD_rate   = 5; 				// 57600
	conf->hex_case    = 0; 				// lower case
	conf->read_base   = HEX;
	conf->pgen_clock  = 0; 				// 1MHz
	conf->opamp       = 0;              // disabled
	
	conf->new_build_marker = CONF_BUILDMARKER;
}

////////////////////////////////////
void get_config(ty_config* conf)
{
	int lb_func = 0;
	while(LINEBREAK_FUNCS[lb_func++] != line_break_func);
	
	int trigger_out_type = 0; // word
	if(isr_edge_out_func == isr_edge_out_lowbyte){
		trigger_out_type = 1;
	}else if(isr_edge_out_func == isr_edge_out_highbyte){
		trigger_out_type = 2;
	}
	
	// ?: used to make sure values are just 0 & 1
	conf->echo        = ( g_echo == 0 ? 0 : 1 );		// *X0
	conf->line_break  = lb_func - 1;					// *X1
	conf->BAUD_rate   = g_baud_rate;					// *X2
	conf->hex_case    = ( g_hex_case == 0 ? 0 : 1 );	// *X3
	conf->adc_channel = g_adc_channels;					// *X4
	conf->edge_trig   = trigger_out_type;				// *X5
	conf->pgen_clock  = ( clock_pGen_GetDividerRegister() == 47 ? 0 : 1 );	// *X6
	conf->timestamp   = g_timestamp;					// *X7
	conf->read_base   = ( g_output_base == 0 ? 0 : 1 );	// *X8
	conf->opamp       = ( OPAMP_POWER_MODE == 0 ? 0 : 1 );
}

////////////////////////////////////
void send_config()
{
	ty_config conf = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	get_config(&conf);
	
	int dig;
	int anl;
	
	get_trig_readout(&anl, &dig);
	
	/* output format 
		!<Version>,<DReg>,<DCache>,<P>,<L>,<*D>,<*O>,<*P>,<*L>,<*T>,<*t>,<*E>
		<~a>,<~b>,<~P>,<*B>,
		<*X0>,<*X1>,<*X2>,<*X3>,<*X4>,<*X5>,<*X6>,<*X7>,<*X8>,<*X9>
	*/
	
	_PF("!%02X%02X,", GPIO232_16_VER_MAJOR, GPIO232_16_VER_MINOR);
	
	_PF("%04X,%04X,%04X,%04X,",
				read_low_high_DR(),
				get_data_cache(),
				pGen_ReadPeriod(),
				pGen_ReadCompare()
				);
				
	_PF("%04X,%04X,%04X,%04X,",
				g_in_out_mask, g_out_mode_mask, g_pull_up_mask, g_led_mode_mask);
	
	_PF("%04X,%04X,0%d%d%d,",
				g_trig_rising,
				g_trig_falling,
				get_ext_trig_edge(),
				dig, anl);
	
	_PF("%02X,%02X,%c%c,",
				( IDAC_1_IDAC_CONTROL_REG & (uint32)IDAC_1_IDAC_VALUE_MASK ),
				( IDAC_2_IDAC_CONTROL_REG & (uint32)IDAC_2_IDAC_VALUE_MASK ),
				( DAC1_POL_BIT ? '0' : '1' ), ( DAC2_POL_BIT ? '0' : '1' ));
	
	_PF("%c,%d,%d,%d,",
				(g_boolfunc_byte ? '1' : '0'),
				conf.echo,
				conf.line_break,
				conf.BAUD_rate);
	
	_PF("%d,%d,%d,%d,%d,%d,%d\r\n",
				conf.hex_case,
				conf.adc_channel,
				conf.edge_trig,
				conf.pgen_clock,
				conf.timestamp,
				conf.read_base,
				conf.opamp);
}

////////////////////////////////////
void myprintf(const char* format, ...)
{
	va_list args;
	char buf[128];
	va_start(args, format);
	int ret = vsprintf(buf, format, args);
	if(ret > 0)UART_1_UartPutString(buf);
	va_end(args);
}

////////////////////////////////////
void print_status_line(const char* name, const uint8 lo, const uint8 hi, const char* line_end)
{
	const uint8 chl = ((lo >= ' ' && lo <= '~') ? lo : ' ');
	const uint8 chh = ((hi >= ' ' && hi <= '~') ? hi : ' ');
	
	myprintf("  %-12s%02x%02x  '%c%c'  ", name, hi, lo, chh, chl);
	printbinary(hi); _PC(' '); printbinary(lo);
	myprintf("       %3d %3d %5d%s", hi, lo, ((uint16)hi << 8) | lo, line_end);
}

////////////////////////////////////
void show_status(int short_list)
{
	const uint16 period     = pGen_ReadPeriod();
	const uint16 pulselen   = pGen_ReadCompare();
	const uint16 data_cache = get_data_cache();
	const uint16 adc1       = ADC_1_GetResult16(0);
	const uint16 adc2       = ADC_1_GetResult16(1);
	
	const uint8 dac1 = ( IDAC_1_IDAC_CONTROL_REG & (uint32)IDAC_1_IDAC_VALUE_MASK );
	const uint8 dac2 = ( IDAC_2_IDAC_CONTROL_REG & (uint32)IDAC_2_IDAC_VALUE_MASK );
	const uint8 pgen = clock_pGen_GetDividerRegister() == 47;

	_PS("\r\n");
	_PS("_ port __________________________________________________________\r\n");
	_PS("              hex   ascii high byte low byte   dec: high low word\r\n");
	_PS("              ----- ----- --------- ---------- ------------------\r\n");
	
	print_status_line("pin state",  IO_low_Read(),           IO_high_Read(), "\r\n");
	print_status_line("outp latch", IO_low_DR,               IO_high_DR, "\r\n");
	print_status_line("outp cache", LOBYTE(data_cache),      HIBYTE(data_cache), "\r\n");
	print_status_line("direction",  LOBYTE(g_in_out_mask),   HIBYTE(g_in_out_mask), "\r\n");
	print_status_line("outp mode",  LOBYTE(g_out_mode_mask), HIBYTE(g_out_mode_mask), "\r\n");
	print_status_line("pull up",    LOBYTE(g_pull_up_mask),  HIBYTE(g_pull_up_mask), "\r\n");
	print_status_line("LED mode",   LOBYTE(g_led_mode_mask), HIBYTE(g_led_mode_mask), "\r\n");
	print_status_line("trig rise",  LOBYTE(g_trig_rising),   HIBYTE(g_trig_rising), "\r\n");
	print_status_line("trig fall",  LOBYTE(g_trig_falling),  HIBYTE(g_trig_falling), "\r\n");
	
	_PS("              ----- ----- --------- ---------- ------------------\r\n");
	print_status_line("PG period",  LOBYTE(period),          HIBYTE(period), "");
		_PF(" %s\r\n", pgen ? "us" : "ms");
	print_status_line("PG length",  LOBYTE(pulselen),        HIBYTE(pulselen), "");
		_PF(" %s\r\n", pgen ? "us" : "ms");
	
	if(pulselen >= period){
		_PS("                                       !!! pulse width >= period !!!\r\n");
	}else{
		_PS("\r\n");
	}
	
	_PS("_ analog units _____________________________________________________\r\n");
	_PF("  DAC 1 : 0x%02X (%3d) / polarity: %c\r\n", dac1, dac1, (DAC1_POL_BIT ? '-' : '+'));
	_PF("  DAC 2 : 0x%02X (%3d) / polarity: %c / buffer opamp: %s\r\n",
		dac2, dac2, ( DAC2_POL_BIT ? '-' : '+' ), (OPAMP_POWER_MODE == 0 ? "off" : "on"));
	_PF("  ADC 1 : 0x%04X (%5d / %4d mV) / ADC 2 : 0x%04X (%5d / %4d mV)\r\n",
		adc1, adc1, adc1 / 2, adc2, adc2, adc2 / 2);
	
	if(short_list == 0){
		_PS("\r\n");
		_PS("_ raw registers ___________________________________\r\n");
		_PF("  drive mode     high: 0x%08X  low: 0x%08X\r\n", IO_high_PC, IO_low_PC);
		_PF("  data register  high: 0x%08X  low: 0x%08X\r\n", IO_high_DR, IO_low_DR);
		_PF("  pin state      high: 0x%08X  low: 0x%08X\r\n", IO_high_PS, IO_low_PS);
		
		_PS("\r\n");
		_PS("_ drive modes ___________________________________________________________\r\n");
		_PS("  port low  : ");
		
		int i=8;
		while(i--) myprintf("%s%c", DRIVEMODE_NAMES[(IO_low_PC >> (3*i)) & 7], i==4 ? '.' : ' ');
		
		_PS("\r\n  port high : ");
		i=8;
		while(i--) myprintf("%s%c", DRIVEMODE_NAMES[(IO_high_PC >> (3*i)) & 7], i==4 ? '.' : ' ');
		
		_PS("\r\n\r\n");
		
		_PS("_ clock settings ________________________________\r\n");
		_PF("  UART      : divider: %d, fraction: %d, oversampling: %d\r\n",
			UART_1_SCBCLK_GetDividerRegister() + 1,
			UART_1_SCBCLK_GetFractionalDividerRegister(),
			(UART_1_CTRL_REG & UART_1_CTRL_OVS_MASK) + 1
		);
		_PF("  Pulse Gen : divider: %d, fraction: %d, clock: %s, running: %s\r\n",
			clock_pGen_GetDividerRegister() + 1,
			clock_pGen_GetFractionalDividerRegister(),
			(pgen ? "1MHz (1us/bit)" : "1kHz (1ms/bit)"),
			((pGen_BLOCK_CONTROL_REG & pGen_MASK) == 0 ? "no" : "yes")
		);
	}
	
	UART_1_UartPutChar('\x1a'); // CTRL-Z to help with cmd.exe tests
}

////////////////////////////////////
void init_readout(int wd, cyisraddress isr)
{
	// 1st: stop all, even if not running
	int_read_timer_Stop();
	read_timer_Stop();

	if(wd == 0) return;
	
	// start timer and interrupt
	read_timer_ClearInterrupt(read_timer_TC_INTERRUPT_MASK);
	int_read_timer_ClearPending();
	
	read_timer_Start();
	read_timer_WriteCounter(0);
	read_timer_WritePeriod(wd);

	int_read_timer_StartEx(isr);
}

////////////////////////////////////
// helper for usage();
int pause_query(){
	_PS(">>> press a key to continue!");
	uint8 ch = UART_getc();
	line_break_CRLF();
	if(ch == 'q' || ch == 'Q' || ch == 27 /* ESC */) return -1;
	return 0;
}

////////////////////////////////////
void usage(int pagewise)
{
	uint8 ch = ' ';
	if(pagewise){
		_PS("select screen height: 24 lines: press 1 / 40 lines: press 2\r\n");
		ch = UART_getc();
		if(ch == 'q' || ch == 'Q' || ch == 27 /* ESC */) return;
	}
	
	_PS("\r\n== GPIO232-16 | by pleXus (2018) | " GPIO232_16_VERSION " ==\r\n");
	_PS("<nn>   = one HEX byte (lower case letters !!)\r\n");
	_PS("<nnnn> = two HEX bytes (one HEX word)\r\n");
	_PS("! for infos on decimal input, hit *?\r\n");
	_PS("-------------------------------\r\n");
	_PS("l<nn>    - set low byte  (port 0)\r\n");
	_PS("h<nn>    - set high byte (port 2)\r\n");
	_PS("w<nnnn>  - set word\r\n");
	_PS("a<nn>    - port AND byte -> port\r\n");
	_PS("o<nn>    - port OR  byte -> port\r\n");
	_PS("x<nn>    - port XOR byte -> port\r\n");
	_PS("A<nnnn>  - port AND word -> port\r\n");
	_PS("O<nnnn>  - port OR  word -> port\r\n");
	_PS("X<nnnn>  - port XOR word -> port\r\n");
	_PS("c        - clear low byte\r\n");
	_PS("C        - clear high byte\r\n");
	_PS("F        - clear word (flush)\r\n");
	_PS("s<n>     - set bit, bit number as one hex digit (0..f)\r\n");
	_PS("z<n>     - clear bit (zero)\r\n");
	_PS("i        - invert low byte\r\n");
	_PS("I        - invert high byte\r\n");
	_PS("N        - invert word (negate)\r\n");
	_PS("r        - read low byte; returns 2 HEX digits \"hh\"\r\n");
	
	// 1st stop in 24 lines mode
	if(pagewise && ch == '1') if(pause_query() == -1) return;
	
	_PS("R        - read high byte\r\n");
	_PS("D        - read word (dump); returns 4 HEX digits \"hhhh\" (MSB left)\r\n");
	_PS("S        - read ADC value (pin 1.4 & 1.5), result depends on *X4\r\n");
	_PS("P<nnnn>  - set pulse generator period, 0000 disables pulse generator\r\n");
	_PS("L<nnnn>  - set pulse width (length)  !! Pulse Generator on Pin 3.7 !!\r\n");
	_PS("M        - PIO monitor on/off (edge triggered)\r\n");
	_PS("[SPACE]  - show register status (use 'q' to get shorter version)\r\n");
	_PS("?        - this help page\r\n");
	_PS("#        - convert decimal number to hex\r\n");
	_PS(".        - cancel transmission, acknowledged by '#'\r\n");
	_PS("[ESC]    - cancels all automatic readouts (kinda buggy)\r\n");
	_PS("b<nn><...> - send low byte burst, <nn> is number of bytes in burst\r\n");
	_PS("             <...> = 2byte hex values, no separator: b<nn><d0><d1>...<dn>\r\n");
	_PS("B<nn><...> - send high byte burst;   bursted data is not cached\r\n");
	_PS("W<nn><...> - send word burst, <nn> is number of words (4byte hex value)\r\n");
	_PS("-------------------------------\r\n");
	
	// 1st stop in 40 lines mode
	if(pagewise && ch == '2') if(pause_query() == -1) return;
	
	_PS("~a<nn>   - set value for DAC 1 (pin 4.0)\r\n");
	_PS("~b<nn>   - set value for DAC 2 (pin 5.0, buffered on 5.2 (see *?)\r\n");
	_PS("~P<nn>   - set DAC polarity; 0=negative (sink), 1=positive (source)\r\n");
	_PS("           each digit for one DAC: <ab>, (<10> -> (DAC1+, DAC2-))\r\n");
	_PS("~r<nnnn> - read ADC value repeatedly every <nnnn> msec\r\n");
	_PS("~M       - monitor ADCs (pin 1.4 & 1.5) (50ms Interval)\r\n");
	_PS("-------------------------------\r\n");
	
	// 2nd stop in 24 lines mode
	if(pagewise && ch == '1') if(pause_query() == -1) return;

	_PS("*D<nnnn> - set pin direction (1=in, 0=out; default 0xffff)\r\n");
	_PS("*O<nnnn> - set output type (0=totem-pole, 1=open drain; default 0000)\r\n");
	_PS("*P<nnnn> - enable strong pull-up (~5k, can drive LEDs),\r\n");
	_PS("           0=off, 1=on, default 0000, ignored if output is totem-pole\r\n");
	_PS("*L<nnnn> - LED mode (can also be used as input to read simple switches)\r\n");
	_PS("           overrides all other port settings! 0=on, 1=off, default 0000\r\n");
	_PS("*T<nnnn> - trigger readout on rising edge @ specific pin,\r\n");
	_PS("           0=trigger off, 1=trigger readout\r\n");
	_PS("*t<nnnn> - trigger readout on falling edge @ specific pin\r\n");
	_PS("*r<nnnn> - read low byte repeatedly every <nnnn> msec,\r\n");
	_PS("           0000 disables repetitive readout\r\n");
	_PS("*R<nnnn> - read high byte repeatedly\r\n");
	_PS("*W<nnnn> - read word repeatedly\r\n");
	_PS("*B<x>    - boolean byte function for:\r\n");
	_PS("           x==0 -> low byte (default) / x!=0 -> high byte\r\n");
	_PS("*E<nnnn> - enable external triggering (pin 3.6)\r\n");
	_PS("           <nnnn> means: <*eda> -> e(egde), d(digital), a(adc)\r\n");
	_PS("           trigger edge,    e: 0=off, 1=rising, 2=falling, 3=both\r\n");
	_PS("           digital readout, d: 0=none, 1=low, 2=high, 3=word\r\n");
	_PS("           analog readout,  a: 0=none, 1=adc1, 2=adc2, 3=adc1:adc2\r\n");
	_PS("*?       - paged help page & some extra information\r\n");
	_PS("*V       - get FW version; answer: \"!<version>\"\r\n");
	_PS("*p       - ping; answer: \"!PONG\"\r\n");
	
	// 3rd stop in 24 lines mode
	if(pagewise && ch == '1') if(pause_query() == -1) return;

	_PS("*Y42     - reset device\r\n");
	_PS("*Y99     - restore factory settings (*X<x> settings), requires reset\r\n");
	_PS("*C       - read config, format:\r\n");
	_PS("           !<Version>,<DReg><DCache>,<P>,<L>,<*D>,<*O>,<*P>,<*L>,\r\n");
	_PS("           <*T>,<*t>,<~a>,<~b>,<~P>,<*B>,<*X0>,<*X1>,<*X2>,<*X3>,<*X4>,\r\n");
	_PS("           <*X5>,<*X6>,<*X7>,<*X8>,<*X9>\\r\\n\r\n");
	_PS("           <DReg> = port data register (output latch, not pin state)\r\n");
	_PS("           <DCache> = port data cache (last value written)\r\n");
	_PS("------------------------------\r\n");

	// 2nd stop in 40 lines mode
	if(pagewise && ch == '2') if(pause_query() == -1) return;

	_PS(" system config commands:   (will be stored nonvolatile)\r\n");
	_PS("*X0<x>   - console echo; x==0 -> off; x!=0 -> on\r\n");
	_PS("*X1<x>   - append line break or separator after read\r\n");
	_PS("           x: 0=off 1=\\n 2=\\r 3=\\r\\n (default) 4=' ' 5=, 6=; 7=/ 8=Tab\r\n");
	_PS("*X2<x>   - set BAUD rate: x: 0=2400, 1=4800, 2=9600, 3=19200, 4=38400,\r\n");
	_PS("                             5=57600 (default), 6=115200, 7=230400\r\n");
	_PS("*X3<x>   - case of hex letters; x: 0=lower case, 1=upper case\r\n");
	_PS("*X4<x>   - set channel for ADC readout; x: 0=Ch1, 1=Ch2, 2=Ch1 & Ch2\r\n");
	_PS("           output on dual-channel: <nnnn>:<mmmm> (n=ch1, m=ch2)\r\n");
	_PS("*X5<x>   - edge triggered readout; x: 0=word, 1=low byte, 2=high byte\r\n");
	_PS("*X6<x>   - pulse generator main clock frequency; x: 0=1MHz (default), 1=1kHz\r\n");
	_PS("*X7<x>   - timestamp format; x: 0=no timestamp, 1='HHMMSS-', 2='MMDD.HHMMSS-'\r\n");
	_PS("           NOTE: lowers performance, always starts at 01.01.1970/00:00:00\r\n");
	_PS("*X8<x>   - default readout base: x: 0=HEX (default), 1=DEC\r\n");
	
	// 4th stop in 24 lines mode
	if(pagewise && ch == '1') if(pause_query() == -1) return;
	
	_PS("*X9<x>   - enable opamp; x: 0=disabled (default), 1=enabled (pin 5.2)\r\n");
	if( ! pagewise) _PS("~~ too many lines? can't scroll? try: *?\r\n");
	
	if(pagewise){
		_PS(" +--------------------------------------------------------------------------+\r\n");
		_PS(" | Decimal Input: Enter decimal number then hit the command character.      |\r\n");
		_PS(" |    e.g.: '1234w' is the same as 'w04d2'; e.g.: '200~a' to set DAC 1      |\r\n");
		_PS(" |    dec input available for: l,h,w,a,o,x,A,O,X,s,z,P,L,~a,~b              |\r\n");
		_PS(" +--------------------------+-----------------------------------------------+\r\n");
		_PS(" | p0.0 - p0.7  low byte    | IO ports Iout max: 10mA                       |\r\n");
		_PS(" | p2.0 - p2.7 high byte    | Pull-Up resistors ~5.6k                       |\r\n");
		_PS(" |                          | LED mode uses pull-up /pull-down resistors to |\r\n");
		_PS(" | p1.4 ADC 1 (12bit)       | drive LEDs directly (to GND or VDD).          |\r\n");
		_PS(" | p1.5 ADC 2 (12bit)       | Current DAC Output 2.4uA/bit                  |\r\n");
		_PS(" |                          | ADC resolution:    500mV/bit                  |\r\n");
		_PS(" | p3.6 external trigger in | Buffered output only works with load resistor |\r\n");
		_PS(" | p3.7 pulse generator     | from p5.0 to GND. OpAmp has to be enabled.    |\r\n");
		_PS(" |                          | Out: ~10mV/bit with 4.2k load res.            |\r\n");
		_PS(" | p4.0 DAC 1 (8bit)        | Iout max: ~5mA                                |\r\n");
		_PS(" | p5.0 DAC 2 (8bit)        | Pulse generator frequency range @ 1MHz clock: |\r\n");
		_PS(" | p5.2 buffered DAC 2 out  | 15.3Hz @ Pffff to 500kHz @ P0001 (L0000)      |\r\n");
		_PS(" |                          | Shortest pulse width: 1us @ L0000             |\r\n");
		_PS(" | p1.1 RS232 TX            | Frequency range @ 1kHz: 15mHz .. 500Hz        |\r\n");
		_PS(" | p1.0 RS232 RX            | shortest pulse: 1ms                           |\r\n");
		_PS(" +--------------------------+-----------------------------------------------+\r\n");
	}
}


////////////////////////////////////
const char DRIVEMODE_NAMES[8][8] = {
	"ALG_HIZ", "DIG_HIZ", "RES_UP ", "RES_DWN", "OD_LO  ", "OD_HI  ", "STRONG ", "RES_U_D"
};

/* [] END OF FILE */
