/*****************************************************************************
* 
*  This file is part of GPIO232-16.
*  
*  GPIO232-16 is free software: you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation, either version 3 of the License, or
*  (at your option) any later version.
*  
*  GPIO232-16 is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*  
*  You should have received a copy of the GNU General Public License
*  along with GPIO232-16.  If not, see <http://www.gnu.org/licenses/>.
*
 ****************************************************************************/
 
#include <project.h>
#include "main.h"
#include "interrupt.h"
#include "io.h"


////////////////////////////////////
void  (*get_line_break_func(int func))(void)
{			
	switch(func){
		case 1: return line_break_LF;
		case 2: return line_break_CR;
		case 3: return line_break_CRLF;
		case 4: return line_break_SPACE;
		case 5: return line_break_COMMA;
		case 6: return line_break_SEMICL;
		case 7: return line_break_DASH;
		case 8: return line_break_TAB;
	}
	
	return line_break_NONE;
}

////////////////////////////////////
void byte_out(uint8 value)
{
	if(g_output_base == HEX){
		put_hex(value);
		return;
	}
	put_dec(value, 1);
}

////////////////////////////////////
void word_out(uint16 value)
{
	if(g_output_base == HEX){
		put_hex(HIBYTE(value));
		put_hex(LOBYTE(value));
		return;
	}
	put_dec(value, 1);
}

// blocking function
char UART_getc()
{
	char ch;
	while((ch = UART_1_UartGetChar()) == 0);
	if(g_echo) UART_1_UartPutChar(ch);
	return ch;
}

// blocking function, never echoes
char UART_getc_ne()
{
	char ch;
	while((ch = UART_1_UartGetChar()) == 0);
	return ch;
}

////////////////////////////////////
int getbyte()
{
	int val = 0;
	int i = 0;
	for(; i<2; i++){
		uint8 ch = UART_getc();
		if(ch == '.') return -1;
		if(ch > '9')
			ch -= 'a' - 10;
		else
			ch -= '0';
		val = (val << 4) + ch;
	}
	return val;
}

////////////////////////////////////
int getword()
{
	int val = 0;
	int i = 0;
	for(; i<4; i++){
		uint8 ch = UART_getc();
		if(ch == '.') return -1;
		if(ch > '9')
			ch -= 'a' - 10;
		else
			ch -= '0';
		val = (val << 4) + ch;
	}
	return val;
}

////////////////////////////////////
void line_break_NONE(){}
void line_break_CR(){ UART_1_UartPutChar('\r'); }
void line_break_LF(){ UART_1_UartPutChar('\n'); }
void line_break_CRLF(){
	// probably faster than sending a string: "\r\n"
	UART_1_UartPutChar('\r');
	UART_1_UartPutChar('\n');
}
void line_break_SPACE(){ UART_1_UartPutChar(' '); }
void line_break_COMMA(){ UART_1_UartPutChar(','); }
void line_break_SEMICL(){ UART_1_UartPutChar(';'); }
void line_break_DASH(){ UART_1_UartPutChar('/'); }
void line_break_TAB(){ UART_1_UartPutChar('\t'); }


////////////////////////////////////
void printbinary(uint8 byteval)
{
	int i = 8;
	while(i--){
		UART_1_UartPutChar( ((byteval >> i) & 1) + '0' );
		if(i==4) UART_1_UartPutChar('.');
	}
}

////////////////////////////////////
void set_baud_rate(uint8 rate_id)
{
	if(rate_id > 7) return;
	g_baud_rate = rate_id;
	UART_1_Stop();
	UART_1_SCBCLK_SetFractionalDividerRegister(UART_SPEED_DEVIDERS[rate_id], 0u);
	UART_1_Start();
}


////////////////////////////////////
void set_ext_trig(uint16 settings)
{
	set_ext_trig_edge((settings >> 8) & 3u);
	set_trig_readout(
		settings & 3u,			// analog readout settings
		(settings >> 4) & 3u	// digital readout settings
	);
}

////////////////////////////////////
void set_ext_trig_edge(uint8 type)
{
	EXT_TRIG_INTCONF = (EXT_TRIG_INTCONF & ~(3u << 12)) | ((uint32)type <<  12);
}

////////////////////////////////////
uint8 get_ext_trig_edge()
{
	return (EXT_TRIG_INTCONF >> 12) & 3u;
}

////////////////////////////////////
void send_timestamp()
{
	if(g_timestamp == 0) return;
	uint32 t = RTC_GetTime();
	
	if(g_timestamp == 2){
		uint32 d = RTC_GetDate();
		put_dec(RTC_GetMonth(d), 2);
		put_dec(RTC_GetDay(d), 2);
		UART_1_UartPutChar('.');
	}
	
	put_dec(RTC_GetHours(t), 2);
	put_dec(RTC_GetMinutes(t), 2);
	put_dec(RTC_GetSecond(t), 2);
	UART_1_UartPutChar('-');
}



////////////////////////////////////
void put_dec(uint16 value, int min_digits)
{
	if(value > 0 || min_digits > 0){
		put_dec(value / 10, min_digits - 1);
		UART_1_UartPutChar(value % 10 + '0');
	}
}


////////////////////////////////////
const char HEX_DIGITS[] = "0123456789abcdef0123456789ABCDEF\0\0";



////////////////////////////////////
const uint16 UART_SPEED_DEVIDERS[] = {
	1250,  // 2400	
	 625,  // 4800
	 313,  // 9600
	 156,  // 19200
	  77,  // 38400
	  52,  // 57600
	  25,  // 115200
	  12   // 230400
};

const cyisraddress LINEBREAK_FUNCS[9] = {
	line_break_NONE,
	line_break_CR,
	line_break_LF,
	line_break_CRLF,
	line_break_SPACE,
	line_break_COMMA,
	line_break_SEMICL,
	line_break_DASH,
	line_break_TAB
};



