/*****************************************************************************
* 
*  This file is part of GPIO232-16.
*  
*  GPIO232-16 is free software: you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation, either version 3 of the License, or
*  (at your option) any later version.
*  
*  GPIO232-16 is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*  
*  You should have received a copy of the GNU General Public License
*  along with GPIO232-16.  If not, see <http://www.gnu.org/licenses/>.
*
 ****************************************************************************/

#include <project.h>
#include "main.h"
#include "interrupt.h"
#include "digital.h"
#include "io.h"

////////////////////////////////////
int l_trig_readout_digital;
int l_trig_readout_analog;


////////////////////////////////////
void (*get_isr_edge_func(int func))(void)
{
	switch(func){
		case 1: return isr_edge_out_lowbyte; 
		case 2: return isr_edge_out_highbyte;
	}
	return isr_edge_out_word;
}

////////////////////////////////////
// external trigger
void set_trig_readout(int analog, int digital)
{
	l_trig_readout_digital = digital;
	l_trig_readout_analog = analog;
}

////////////////////////////////////
// external trigger
void get_trig_readout(int* analog, int* digital)
{
	*analog = l_trig_readout_analog;
	*digital = l_trig_readout_digital;
}

////////////////////////////////////
void isr_ext_trig()
{
	static uint16 res;
	
	if(l_trig_readout_digital == 0 && l_trig_readout_analog == 0){
		ext_trig_ClearInterrupt();
		return;
	}
	
	send_timestamp();
	
	if(l_trig_readout_digital == EXT_TRIG_READOUT_DIG_LOW){
		byte_out(IO_low_Read());
	}else if(l_trig_readout_digital == EXT_TRIG_READOUT_DIG_HIGH){
		byte_out(IO_high_Read());
	}else if(l_trig_readout_digital == EXT_TRIG_READOUT_DIG_WORD){
		word_out(MAKEWORD(IO_low_Read(), IO_high_Read()));
	}
	
	// insert special separator if both readout types are active
	if(l_trig_readout_digital != 0 && l_trig_readout_analog != 0) UART_1_UartPutChar('|');
	
	// using '&' here, because only bits 0 and 1 are used; easy to select that way
	if(l_trig_readout_analog & EXT_TRIG_READOUT_ANA_ADC1){
		res = ADC_1_GetResult16(0);
		word_out(res);
	}
	
	if(l_trig_readout_analog & EXT_TRIG_READOUT_ANA_ADC2){
		// insert separator if both ADCs are active
		if(l_trig_readout_analog & EXT_TRIG_READOUT_ANA_ADC1) UART_1_UartPutChar(':');
		res = ADC_1_GetResult16(1);
		word_out(res);
	}
	
	line_break_func();
		
	ext_trig_ClearInterrupt();
}


////////////////////////////////////
void isr_edge_low(void)
{
	static union ty_instat intreg;
	static uint8 stat;
	
	// shove the value into an enum to get easier access to the fields
	// hopefully, faster
	intreg.reg = IO_low_INTSTAT;
	
	if(
		(intreg.int_flag & intreg.pin_state & LOBYTE(g_trig_rising)) != 0
		||
		(intreg.int_flag & (~intreg.pin_state) & LOBYTE(g_trig_falling)) != 0
	){
		stat = CyEnterCriticalSection();
		isr_edge_out_func();
		CyExitCriticalSection(stat);
	}
	IO_low_ClearInterrupt();
}

////////////////////////////////////
void isr_edge_high(void)
{
	static union ty_instat intreg;
	static uint8 stat;
	
	intreg.reg = IO_high_INTSTAT;
	
	if(
		(intreg.int_flag & intreg.pin_state & HIBYTE(g_trig_rising)) != 0
		||
		(intreg.int_flag & (~intreg.pin_state) & HIBYTE(g_trig_falling)) != 0
	){
		stat = CyEnterCriticalSection();
		isr_edge_out_func();
		CyExitCriticalSection(stat);
	}
	IO_high_ClearInterrupt();
}

void isr_edge_out_lowbyte(void)
{
	send_timestamp();
	byte_out(IO_low_Read());
	line_break_func();
}
void isr_edge_out_highbyte(void)
{
	send_timestamp();
	byte_out(IO_high_Read());
	line_break_func();
}
void isr_edge_out_word(void)
{
	send_timestamp();
	word_out(MAKEWORD(IO_low_Read(), IO_high_Read()));
	line_break_func();
}

////////////////////////////////////
void isr_read_timer_low(void)
{
	send_timestamp();
	byte_out(IO_low_Read());
	line_break_func();
	read_timer_ClearInterrupt(read_timer_TC_INTERRUPT_MASK);
}
void isr_read_timer_high(void)
{
	send_timestamp();
	byte_out(IO_high_Read());
	line_break_func();
	read_timer_ClearInterrupt(read_timer_TC_INTERRUPT_MASK);
}
void isr_read_timer_word(void)
{
	send_timestamp();
	word_out(MAKEWORD(IO_low_Read(), IO_high_Read()));
	line_break_func();
	read_timer_ClearInterrupt(read_timer_TC_INTERRUPT_MASK);
}

////////////////////////////////////
void isr_read_timer_adc_single(void)
{
	static int16 adc_res;
	
	send_timestamp();
	adc_res = ADC_1_GetResult16(g_adc_channels);
	word_out(adc_res);
	line_break_func();
	
	read_timer_ClearInterrupt(read_timer_TC_INTERRUPT_MASK);
}

////////////////////////////////////
void isr_read_timer_adc_dual(void)
{
	static int16 adc_res;
	
	send_timestamp();
	adc_res = ADC_1_GetResult16(0);
	word_out(adc_res);
	
	UART_1_UartPutChar(':');
	
	adc_res = ADC_1_GetResult16(1);
	word_out(adc_res);
	
	line_break_func();

	read_timer_ClearInterrupt(read_timer_TC_INTERRUPT_MASK);
}

////////////////////////////////////
// edge triggered monitor
void isr_monitor(void)
{
	uint8 stat = CyEnterCriticalSection();
	refresh_monitor();
	CyExitCriticalSection(stat);
}
// entry functions
void isr_monitor_low(void)
{
	isr_monitor();
	IO_low_ClearInterrupt();
}
void isr_monitor_high(void)
{
	isr_monitor();
	IO_high_ClearInterrupt();
}

////////////////////////////////////
void put_adc_monitor_line(uint16 value)
{
	if(value == 0xFFF) UART_1_UartPutChar('*');
	myprintf("0x%04X  %4d  (%4d mV)", value, value, value / 2);
}


////////////////////////////////////
// timer triggered monitor for ADC
void isr_monitor_timer(void)
{
	UART_1_UartPutString("\x1B[2;1H"); // ESC[ sequence, set cursor (1,4)
	
	put_adc_monitor_line(ADC_1_GetResult16(0));
	UART_1_UartPutChar('\r');
	UART_1_UartPutChar('\n');
	put_adc_monitor_line(ADC_1_GetResult16(1));
	
	monitor_timer_ClearInterrupt(monitor_timer_TC_INTERRUPT_MASK);
}





