/*****************************************************************************
* 
*  This file is part of GPIO232-16.
*  
*  GPIO232-16 is free software: you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation, either version 3 of the License, or
*  (at your option) any later version.
*  
*  GPIO232-16 is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*  
*  You should have received a copy of the GNU General Public License
*  along with GPIO232-16.  If not, see <http://www.gnu.org/licenses/>.
*
 ****************************************************************************/
 
#include <project.h>
#include "main.h"
#include "interrupt.h"
#include "digital.h"
#include "io.h"

////////////////////////////////////
uint16 l_last_port_data;


////////////////////////////////////
uint16 get_data_cache(){ return l_last_port_data; }

////////////////////////////////////
void set_pGen(uint16 period, uint16 pulsewidth)
{
	if(period == 0){
		pGen_Stop();
		clock_RSFF_Stop();
		clock_pGen_Stop();
		// write values even on stop,
		// b/c the status page reads from the registers
		pGen_WritePeriod(period);
		pGen_WriteCompare(pulsewidth);
		return;
	}
	
	clock_RSFF_Start();
	clock_pGen_Start();

	pGen_Start();
	
	pGen_WriteCounter(0);
	pGen_WritePeriod(period);
	pGen_WriteCompare(pulsewidth);
}


////////////////////////////////////
void init_gpio()
{
	g_in_out_mask    = 0xFFFF; // all input (Hi-Z)
	g_out_mode_mask  = 0x0000; // all totem pole
	g_pull_up_mask   = 0x0000; // all pull ups off
	g_led_mode_mask  = 0x0000; // all LED mode off
	l_last_port_data = 0;
	
	set_drive_mode();
	
	// initialize pulse generator to zero
	pGen_WritePeriod(0);
	pGen_WriteCompare(0);
}

////////////////////////////////////
int send_burst_byte(char ch, uint8 len)
{
	void (*out_func)(uint8);
	out_func = (ch == 'b' ? set_low_port : set_high_port );
	
	while(len--){
		int bt = getbyte();
		if(bt < 0) return -1;
		out_func(bt);
	}

	return 0;
}

////////////////////////////////////
int send_burst_word(uint8 len)
{
	while(len--){
		int wd = getword();
		if(wd < 0) return -1;
		set_word_port(wd);
	}
	return 0;
}

////////////////////////////////////
void init_monitor_mode()
{
	int_edge_low_Stop();
	int_edge_high_Stop();
	read_timer_Stop();

	if(g_monitor_mode == 1){
		g_monitor_mode = 0;
		set_edge_trigger(); // restore edge trigger settings
		UART_1_UartPutChar('!');
		return;
	}
	
	g_monitor_mode = 1;
	uint16 data_cache = get_data_cache();
	
	UART_1_UartPutChar('\x0c'); // ASCII form feed command
	
	_PS("_ edge triggered monitor (send 'M' to exit) ______________________\r\n");
	_PS("              hex   ascii high byte low byte   dec: high low word\r\n");
	_PS("              ----- ----- --------- ---------- -------------------\r\n");
	
	print_status_line("pin state",  IO_LOW_PINSTATE,    IO_HIGH_PINSTATE, "\r\n");
	print_status_line("outp latch", IO_LOW_DATAREG,     IO_HIGH_DATAREG, "\r\n");
	print_status_line("outp cache", LOBYTE(data_cache), HIBYTE(data_cache), "\r\n");
	
	IO_low_ClearInterrupt();
	int_edge_low_StartEx(isr_monitor_low);
	IO_high_ClearInterrupt();
	int_edge_high_StartEx(isr_monitor_high);
}

////////////////////////////////////
void refresh_monitor()
{
	static uint16 data_cache;
	data_cache = get_data_cache();
	
	UART_1_UartPutString("\x1B[4;1H"); // ESC[ sequence, set cursor (1,4)
	print_status_line("pin state",  IO_LOW_PINSTATE, IO_HIGH_PINSTATE, "\r\n");
	print_status_line("outp latch", IO_LOW_DATAREG,  IO_HIGH_DATAREG, "\r\n");
	print_status_line("outp cache", LOBYTE(data_cache), HIBYTE(data_cache), "\r\n");
}

////////////////////////////////////
void set_edge_trigger()
{
	// no flags set for low byte ?
	if(((g_trig_falling | g_trig_rising) & 0x00FF) == 0){
		int_edge_low_Stop();
	}else{
		IO_low_ClearInterrupt();
		int_edge_low_StartEx(isr_edge_low);
	}
	
	// flag check for high byte
	if(((g_trig_falling | g_trig_rising) & 0xFF00) == 0){
		int_edge_high_Stop();
	}else{
		IO_high_ClearInterrupt();
		int_edge_high_StartEx(isr_edge_high);
	}
}


////////////////////////////////////
int boolfunc_byte(char ch, int dec_inp)
{
	int bt = 0;
	
	if(dec_inp < 0){
		bt = getbyte();
		if(bt < 0) return -1;
	}else{
		bt = (uint8)dec_inp;
	}
	
	uint8 port_bt;
	
	if(g_boolfunc_byte == LOW_BYTE){
		port_bt = LOBYTE(get_data_cache());
	}else{
		port_bt = HIBYTE(get_data_cache());
	}
	
	switch(ch){
		case 'a': port_bt &= bt; break;
		case 'o': port_bt |= bt; break;
		case 'x': port_bt ^= bt; break;
	}
	
	if(g_boolfunc_byte == LOW_BYTE){
		set_low_byte(port_bt);
	}else{
		set_high_byte(port_bt);
	}

	return 0;
}

////////////////////////////////////
int boolfunc_word(char ch, int dec_inp)
{
	int wd = 0;
	
	if(dec_inp < 0){
		wd = getword();
		if(wd < 0) return -1;
	}else{
		wd = (uint16)dec_inp;
	}

	uint16 port_wd = get_data_cache();

	switch(ch){
		case 'A': port_wd &= wd; break;
		case 'O': port_wd |= wd; break;
		case 'X': port_wd ^= wd; break;
	}
	
	set_word(port_wd);

	return 0;
}

////////////////////////////////////
void set_low_byte(uint8 bt)
{
	l_last_port_data = (l_last_port_data & 0xFF00) | (uint16)bt;
	set_low_port(bt);
}

////////////////////////////////////
void set_high_byte(uint8 bt)
{
	l_last_port_data = (l_last_port_data & 0x00FF) | ((uint16)bt << 8);
	set_high_port(bt);
}

////////////////////////////////////
void set_word(uint16 wd)
{
	l_last_port_data = wd;
	set_word_port(wd);
}

////////////////////////////////////
void set_low_port(uint8 bt)
{
	// OR in in/out mask to make sure the input pins are '1'
	// important in case pull-ups are enabled
	// mask out led mode pins which shall pass untouched
	IO_low_DR = (uint32)(bt | (uint8)(g_in_out_mask & ~g_led_mode_mask));
}

////////////////////////////////////
void set_high_port(uint8 bt)
{
	IO_high_DR = (uint32)(bt | (uint8)((g_in_out_mask & ~g_led_mode_mask) >> 8));
}

////////////////////////////////////
void set_word_port(uint16 wd)
{
	IO_low_DR = (uint32)(wd | (g_in_out_mask & ~g_led_mode_mask));
	IO_high_DR = (uint32)((wd | (g_in_out_mask & ~g_led_mode_mask)) >> 8);
}

////////////////////////////////////
int set_drive_mode_byte(uint32 reg, int start)
{
	// low byte
	uint32 tmp = *(reg32*)reg; // read register once
	
	int i;
	for(i=start; i<start+8; i++){ // do one byte
	
		if(ISLEDMODE(i))
		{
			CY_SYS_PINS_SET_DRIVE_MODE(&tmp, i-start, CY_SYS_PINS_DM_RES_UPDWN);
		}
		
		else if(ISINPUT(i) && NOPULLUP(i))
		{
			CY_SYS_PINS_SET_DRIVE_MODE(&tmp, i-start, CY_SYS_PINS_DM_DIG_HIZ);
		}
		
		else if(ISINPUT(i) && ISPULLUP(i))
		{
			CY_SYS_PINS_SET_DRIVE_MODE(&tmp, i-start, CY_SYS_PINS_DM_RES_UP);
		}
		
		else if(ISOUTPUT(i) && ISTOTEMPOLE(i))
		{
			CY_SYS_PINS_SET_DRIVE_MODE(&tmp, i-start, CY_SYS_PINS_DM_STRONG);
		}
		
		else if(ISOUTPUT(i) && ISOPENDRAIN(i) && NOPULLUP(i))
		{
			CY_SYS_PINS_SET_DRIVE_MODE(&tmp, i-start, CY_SYS_PINS_DM_OD_LO);
		}
		
		else if(ISOUTPUT(i) && ISOPENDRAIN(i) && ISPULLUP(i))
		{
			CY_SYS_PINS_SET_DRIVE_MODE(&tmp, i-start, CY_SYS_PINS_DM_RES_UP);
		}else{
			// shouldn't happen, if so, escape
			return -1;
		}
	}
	
	*(reg32*)reg = tmp; // write port register once
	
	return 0;
}

////////////////////////////////////
void set_drive_mode()
{
	if(
		set_drive_mode_byte(IO_low__PC, 0) < 0
		||
		set_drive_mode_byte(IO_high__PC, 8) < 0
	){
		// error occured, init to sane values
		g_in_out_mask   = 0xFFFF; // all input (Hi-Z)
		g_out_mode_mask = 0x0000; // all totem pole
		g_pull_up_mask  = 0x0000; // all pull ups off
		g_led_mode_mask = 0x0000; // all LED mode off
		set_drive_mode();
		return;
	}
	set_word(l_last_port_data);
}