/*******************************************************************************
* File Name: timer_ov.c  
* Version 2.10
*
* Description:
*  This file contains API to enable firmware control of a Pins component.
*
* Note:
*
********************************************************************************
* Copyright 2008-2014, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions, 
* disclaimers, and limitations in the end user license agreement accompanying 
* the software package with which this file was provided.
*******************************************************************************/

#include "cytypes.h"
#include "timer_ov.h"

#define SetP4PinDriveMode(shift, mode)  \
    do { \
        timer_ov_PC =   (timer_ov_PC & \
                                (uint32)(~(uint32)(timer_ov_DRIVE_MODE_IND_MASK << (timer_ov_DRIVE_MODE_BITS * (shift))))) | \
                                (uint32)((uint32)(mode) << (timer_ov_DRIVE_MODE_BITS * (shift))); \
    } while (0)


/*******************************************************************************
* Function Name: timer_ov_Write
********************************************************************************
*
* Summary:
*  Assign a new value to the digital port's data output register.  
*
* Parameters:  
*  prtValue:  The value to be assigned to the Digital Port. 
*
* Return: 
*  None 
*  
*******************************************************************************/
void timer_ov_Write(uint8 value) 
{
    uint8 drVal = (uint8)(timer_ov_DR & (uint8)(~timer_ov_MASK));
    drVal = (drVal | ((uint8)(value << timer_ov_SHIFT) & timer_ov_MASK));
    timer_ov_DR = (uint32)drVal;
}


/*******************************************************************************
* Function Name: timer_ov_SetDriveMode
********************************************************************************
*
* Summary:
*  Change the drive mode on the pins of the port.
* 
* Parameters:  
*  mode:  Change the pins to one of the following drive modes.
*
*  timer_ov_DM_STRONG     Strong Drive 
*  timer_ov_DM_OD_HI      Open Drain, Drives High 
*  timer_ov_DM_OD_LO      Open Drain, Drives Low 
*  timer_ov_DM_RES_UP     Resistive Pull Up 
*  timer_ov_DM_RES_DWN    Resistive Pull Down 
*  timer_ov_DM_RES_UPDWN  Resistive Pull Up/Down 
*  timer_ov_DM_DIG_HIZ    High Impedance Digital 
*  timer_ov_DM_ALG_HIZ    High Impedance Analog 
*
* Return: 
*  None
*
*******************************************************************************/
void timer_ov_SetDriveMode(uint8 mode) 
{
	SetP4PinDriveMode(timer_ov__0__SHIFT, mode);
}


/*******************************************************************************
* Function Name: timer_ov_Read
********************************************************************************
*
* Summary:
*  Read the current value on the pins of the Digital Port in right justified 
*  form.
*
* Parameters:  
*  None 
*
* Return: 
*  Returns the current value of the Digital Port as a right justified number
*  
* Note:
*  Macro timer_ov_ReadPS calls this function. 
*  
*******************************************************************************/
uint8 timer_ov_Read(void) 
{
    return (uint8)((timer_ov_PS & timer_ov_MASK) >> timer_ov_SHIFT);
}


/*******************************************************************************
* Function Name: timer_ov_ReadDataReg
********************************************************************************
*
* Summary:
*  Read the current value assigned to a Digital Port's data output register
*
* Parameters:  
*  None 
*
* Return: 
*  Returns the current value assigned to the Digital Port's data output register
*  
*******************************************************************************/
uint8 timer_ov_ReadDataReg(void) 
{
    return (uint8)((timer_ov_DR & timer_ov_MASK) >> timer_ov_SHIFT);
}


/* If Interrupts Are Enabled for this Pins component */ 
#if defined(timer_ov_INTSTAT) 

    /*******************************************************************************
    * Function Name: timer_ov_ClearInterrupt
    ********************************************************************************
    *
    * Summary:
    *  Clears any active interrupts attached to port and returns the value of the 
    *  interrupt status register.
    *
    * Parameters:  
    *  None 
    *
    * Return: 
    *  Returns the value of the interrupt status register
    *  
    *******************************************************************************/
    uint8 timer_ov_ClearInterrupt(void) 
    {
		uint8 maskedStatus = (uint8)(timer_ov_INTSTAT & timer_ov_MASK);
		timer_ov_INTSTAT = maskedStatus;
        return maskedStatus >> timer_ov_SHIFT;
    }

#endif /* If Interrupts Are Enabled for this Pins component */ 


/* [] END OF FILE */
