/*******************************************************************************
* File Name: pGen.c
* Version 2.10
*
* Description:
*  This file provides the source code to the API for the pGen
*  component
*
* Note:
*  None
*
********************************************************************************
* Copyright 2013-2015, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions,
* disclaimers, and limitations in the end user license agreement accompanying
* the software package with which this file was provided.
*******************************************************************************/

#include "pGen.h"

uint8 pGen_initVar = 0u;


/*******************************************************************************
* Function Name: pGen_Init
********************************************************************************
*
* Summary:
*  Initialize/Restore default pGen configuration.
*
* Parameters:
*  None
*
* Return:
*  None
*
*******************************************************************************/
void pGen_Init(void)
{

    /* Set values from customizer to CTRL */
    #if (pGen__QUAD == pGen_CONFIG)
        pGen_CONTROL_REG = pGen_CTRL_QUAD_BASE_CONFIG;
        
        /* Set values from customizer to CTRL1 */
        pGen_TRIG_CONTROL1_REG  = pGen_QUAD_SIGNALS_MODES;

        /* Set values from customizer to INTR */
        pGen_SetInterruptMode(pGen_QUAD_INTERRUPT_MASK);
        
         /* Set other values */
        pGen_SetCounterMode(pGen_COUNT_DOWN);
        pGen_WritePeriod(pGen_QUAD_PERIOD_INIT_VALUE);
        pGen_WriteCounter(pGen_QUAD_PERIOD_INIT_VALUE);
    #endif  /* (pGen__QUAD == pGen_CONFIG) */

    #if (pGen__TIMER == pGen_CONFIG)
        pGen_CONTROL_REG = pGen_CTRL_TIMER_BASE_CONFIG;
        
        /* Set values from customizer to CTRL1 */
        pGen_TRIG_CONTROL1_REG  = pGen_TIMER_SIGNALS_MODES;
    
        /* Set values from customizer to INTR */
        pGen_SetInterruptMode(pGen_TC_INTERRUPT_MASK);
        
        /* Set other values from customizer */
        pGen_WritePeriod(pGen_TC_PERIOD_VALUE );

        #if (pGen__COMPARE == pGen_TC_COMP_CAP_MODE)
            pGen_WriteCompare(pGen_TC_COMPARE_VALUE);

            #if (1u == pGen_TC_COMPARE_SWAP)
                pGen_SetCompareSwap(1u);
                pGen_WriteCompareBuf(pGen_TC_COMPARE_BUF_VALUE);
            #endif  /* (1u == pGen_TC_COMPARE_SWAP) */
        #endif  /* (pGen__COMPARE == pGen_TC_COMP_CAP_MODE) */

        /* Initialize counter value */
        #if (pGen_CY_TCPWM_V2 && pGen_TIMER_UPDOWN_CNT_USED && !pGen_CY_TCPWM_4000)
            pGen_WriteCounter(1u);
        #elif(pGen__COUNT_DOWN == pGen_TC_COUNTER_MODE)
            pGen_WriteCounter(pGen_TC_PERIOD_VALUE);
        #else
            pGen_WriteCounter(0u);
        #endif /* (pGen_CY_TCPWM_V2 && pGen_TIMER_UPDOWN_CNT_USED && !pGen_CY_TCPWM_4000) */
    #endif  /* (pGen__TIMER == pGen_CONFIG) */

    #if (pGen__PWM_SEL == pGen_CONFIG)
        pGen_CONTROL_REG = pGen_CTRL_PWM_BASE_CONFIG;

        #if (pGen__PWM_PR == pGen_PWM_MODE)
            pGen_CONTROL_REG |= pGen_CTRL_PWM_RUN_MODE;
            pGen_WriteCounter(pGen_PWM_PR_INIT_VALUE);
        #else
            pGen_CONTROL_REG |= pGen_CTRL_PWM_ALIGN | pGen_CTRL_PWM_KILL_EVENT;
            
            /* Initialize counter value */
            #if (pGen_CY_TCPWM_V2 && pGen_PWM_UPDOWN_CNT_USED && !pGen_CY_TCPWM_4000)
                pGen_WriteCounter(1u);
            #elif (pGen__RIGHT == pGen_PWM_ALIGN)
                pGen_WriteCounter(pGen_PWM_PERIOD_VALUE);
            #else 
                pGen_WriteCounter(0u);
            #endif  /* (pGen_CY_TCPWM_V2 && pGen_PWM_UPDOWN_CNT_USED && !pGen_CY_TCPWM_4000) */
        #endif  /* (pGen__PWM_PR == pGen_PWM_MODE) */

        #if (pGen__PWM_DT == pGen_PWM_MODE)
            pGen_CONTROL_REG |= pGen_CTRL_PWM_DEAD_TIME_CYCLE;
        #endif  /* (pGen__PWM_DT == pGen_PWM_MODE) */

        #if (pGen__PWM == pGen_PWM_MODE)
            pGen_CONTROL_REG |= pGen_CTRL_PWM_PRESCALER;
        #endif  /* (pGen__PWM == pGen_PWM_MODE) */

        /* Set values from customizer to CTRL1 */
        pGen_TRIG_CONTROL1_REG  = pGen_PWM_SIGNALS_MODES;
    
        /* Set values from customizer to INTR */
        pGen_SetInterruptMode(pGen_PWM_INTERRUPT_MASK);

        /* Set values from customizer to CTRL2 */
        #if (pGen__PWM_PR == pGen_PWM_MODE)
            pGen_TRIG_CONTROL2_REG =
                    (pGen_CC_MATCH_NO_CHANGE    |
                    pGen_OVERLOW_NO_CHANGE      |
                    pGen_UNDERFLOW_NO_CHANGE);
        #else
            #if (pGen__LEFT == pGen_PWM_ALIGN)
                pGen_TRIG_CONTROL2_REG = pGen_PWM_MODE_LEFT;
            #endif  /* ( pGen_PWM_LEFT == pGen_PWM_ALIGN) */

            #if (pGen__RIGHT == pGen_PWM_ALIGN)
                pGen_TRIG_CONTROL2_REG = pGen_PWM_MODE_RIGHT;
            #endif  /* ( pGen_PWM_RIGHT == pGen_PWM_ALIGN) */

            #if (pGen__CENTER == pGen_PWM_ALIGN)
                pGen_TRIG_CONTROL2_REG = pGen_PWM_MODE_CENTER;
            #endif  /* ( pGen_PWM_CENTER == pGen_PWM_ALIGN) */

            #if (pGen__ASYMMETRIC == pGen_PWM_ALIGN)
                pGen_TRIG_CONTROL2_REG = pGen_PWM_MODE_ASYM;
            #endif  /* (pGen__ASYMMETRIC == pGen_PWM_ALIGN) */
        #endif  /* (pGen__PWM_PR == pGen_PWM_MODE) */

        /* Set other values from customizer */
        pGen_WritePeriod(pGen_PWM_PERIOD_VALUE );
        pGen_WriteCompare(pGen_PWM_COMPARE_VALUE);

        #if (1u == pGen_PWM_COMPARE_SWAP)
            pGen_SetCompareSwap(1u);
            pGen_WriteCompareBuf(pGen_PWM_COMPARE_BUF_VALUE);
        #endif  /* (1u == pGen_PWM_COMPARE_SWAP) */

        #if (1u == pGen_PWM_PERIOD_SWAP)
            pGen_SetPeriodSwap(1u);
            pGen_WritePeriodBuf(pGen_PWM_PERIOD_BUF_VALUE);
        #endif  /* (1u == pGen_PWM_PERIOD_SWAP) */
    #endif  /* (pGen__PWM_SEL == pGen_CONFIG) */
    
}


/*******************************************************************************
* Function Name: pGen_Enable
********************************************************************************
*
* Summary:
*  Enables the pGen.
*
* Parameters:
*  None
*
* Return:
*  None
*
*******************************************************************************/
void pGen_Enable(void)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();
    pGen_BLOCK_CONTROL_REG |= pGen_MASK;
    CyExitCriticalSection(enableInterrupts);

    /* Start Timer or PWM if start input is absent */
    #if (pGen__PWM_SEL == pGen_CONFIG)
        #if (0u == pGen_PWM_START_SIGNAL_PRESENT)
            pGen_TriggerCommand(pGen_MASK, pGen_CMD_START);
        #endif /* (0u == pGen_PWM_START_SIGNAL_PRESENT) */
    #endif /* (pGen__PWM_SEL == pGen_CONFIG) */

    #if (pGen__TIMER == pGen_CONFIG)
        #if (0u == pGen_TC_START_SIGNAL_PRESENT)
            pGen_TriggerCommand(pGen_MASK, pGen_CMD_START);
        #endif /* (0u == pGen_TC_START_SIGNAL_PRESENT) */
    #endif /* (pGen__TIMER == pGen_CONFIG) */
    
    #if (pGen__QUAD == pGen_CONFIG)
        #if (0u != pGen_QUAD_AUTO_START)
            pGen_TriggerCommand(pGen_MASK, pGen_CMD_RELOAD);
        #endif /* (0u != pGen_QUAD_AUTO_START) */
    #endif  /* (pGen__QUAD == pGen_CONFIG) */
}


/*******************************************************************************
* Function Name: pGen_Start
********************************************************************************
*
* Summary:
*  Initializes the pGen with default customizer
*  values when called the first time and enables the pGen.
*  For subsequent calls the configuration is left unchanged and the component is
*  just enabled.
*
* Parameters:
*  None
*
* Return:
*  None
*
* Global variables:
*  pGen_initVar: global variable is used to indicate initial
*  configuration of this component.  The variable is initialized to zero and set
*  to 1 the first time pGen_Start() is called. This allows
*  enabling/disabling a component without re-initialization in all subsequent
*  calls to the pGen_Start() routine.
*
*******************************************************************************/
void pGen_Start(void)
{
    if (0u == pGen_initVar)
    {
        pGen_Init();
        pGen_initVar = 1u;
    }

    pGen_Enable();
}


/*******************************************************************************
* Function Name: pGen_Stop
********************************************************************************
*
* Summary:
*  Disables the pGen.
*
* Parameters:
*  None
*
* Return:
*  None
*
*******************************************************************************/
void pGen_Stop(void)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_BLOCK_CONTROL_REG &= (uint32)~pGen_MASK;

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_SetMode
********************************************************************************
*
* Summary:
*  Sets the operation mode of the pGen. This function is used when
*  configured as a generic pGen and the actual mode of operation is
*  set at runtime. The mode must be set while the component is disabled.
*
* Parameters:
*  mode: Mode for the pGen to operate in
*   Values:
*   - pGen_MODE_TIMER_COMPARE - Timer / Counter with
*                                                 compare capability
*         - pGen_MODE_TIMER_CAPTURE - Timer / Counter with
*                                                 capture capability
*         - pGen_MODE_QUAD - Quadrature decoder
*         - pGen_MODE_PWM - PWM
*         - pGen_MODE_PWM_DT - PWM with dead time
*         - pGen_MODE_PWM_PR - PWM with pseudo random capability
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetMode(uint32 mode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_CONTROL_REG &= (uint32)~pGen_MODE_MASK;
    pGen_CONTROL_REG |= mode;

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_SetQDMode
********************************************************************************
*
* Summary:
*  Sets the the Quadrature Decoder to one of the 3 supported modes.
*  Its functionality is only applicable to Quadrature Decoder operation.
*
* Parameters:
*  qdMode: Quadrature Decoder mode
*   Values:
*         - pGen_MODE_X1 - Counts on phi 1 rising
*         - pGen_MODE_X2 - Counts on both edges of phi1 (2x faster)
*         - pGen_MODE_X4 - Counts on both edges of phi1 and phi2
*                                        (4x faster)
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetQDMode(uint32 qdMode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_CONTROL_REG &= (uint32)~pGen_QUAD_MODE_MASK;
    pGen_CONTROL_REG |= qdMode;

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_SetPrescaler
********************************************************************************
*
* Summary:
*  Sets the prescaler value that is applied to the clock input.  Not applicable
*  to a PWM with the dead time mode or Quadrature Decoder mode.
*
* Parameters:
*  prescaler: Prescaler divider value
*   Values:
*         - pGen_PRESCALE_DIVBY1    - Divide by 1 (no prescaling)
*         - pGen_PRESCALE_DIVBY2    - Divide by 2
*         - pGen_PRESCALE_DIVBY4    - Divide by 4
*         - pGen_PRESCALE_DIVBY8    - Divide by 8
*         - pGen_PRESCALE_DIVBY16   - Divide by 16
*         - pGen_PRESCALE_DIVBY32   - Divide by 32
*         - pGen_PRESCALE_DIVBY64   - Divide by 64
*         - pGen_PRESCALE_DIVBY128  - Divide by 128
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetPrescaler(uint32 prescaler)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_CONTROL_REG &= (uint32)~pGen_PRESCALER_MASK;
    pGen_CONTROL_REG |= prescaler;

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_SetOneShot
********************************************************************************
*
* Summary:
*  Writes the register that controls whether the pGen runs
*  continuously or stops when terminal count is reached.  By default the
*  pGen operates in the continuous mode.
*
* Parameters:
*  oneShotEnable
*   Values:
*     - 0 - Continuous
*     - 1 - Enable One Shot
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetOneShot(uint32 oneShotEnable)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_CONTROL_REG &= (uint32)~pGen_ONESHOT_MASK;
    pGen_CONTROL_REG |= ((uint32)((oneShotEnable & pGen_1BIT_MASK) <<
                                                               pGen_ONESHOT_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_SetPWMMode
********************************************************************************
*
* Summary:
*  Writes the control register that determines what mode of operation the PWM
*  output lines are driven in.  There is a setting for what to do on a
*  comparison match (CC_MATCH), on an overflow (OVERFLOW) and on an underflow
*  (UNDERFLOW).  The value for each of the three must be ORed together to form
*  the mode.
*
* Parameters:
*  modeMask: A combination of three mode settings.  Mask must include a value
*  for each of the three or use one of the preconfigured PWM settings.
*   Values:
*     - CC_MATCH_SET        - Set on comparison match
*     - CC_MATCH_CLEAR      - Clear on comparison match
*     - CC_MATCH_INVERT     - Invert on comparison match
*     - CC_MATCH_NO_CHANGE  - No change on comparison match
*     - OVERLOW_SET         - Set on overflow
*     - OVERLOW_CLEAR       - Clear on  overflow
*     - OVERLOW_INVERT      - Invert on overflow
*     - OVERLOW_NO_CHANGE   - No change on overflow
*     - UNDERFLOW_SET       - Set on underflow
*     - UNDERFLOW_CLEAR     - Clear on underflow
*     - UNDERFLOW_INVERT    - Invert on underflow
*     - UNDERFLOW_NO_CHANGE - No change on underflow
*     - PWM_MODE_LEFT       - Setting for left aligned PWM.  Should be combined
*                             with up counting mode
*     - PWM_MODE_RIGHT      - Setting for right aligned PWM.  Should be combined
*                             with down counting mode
*     - PWM_MODE_CENTER     - Setting for center aligned PWM.  Should be
*                             combined with up/down 0 mode
*     - PWM_MODE_ASYM       - Setting for asymmetric PWM.  Should be combined
*                             with up/down 1 mode
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetPWMMode(uint32 modeMask)
{
    pGen_TRIG_CONTROL2_REG = (modeMask & pGen_6BIT_MASK);
}



/*******************************************************************************
* Function Name: pGen_SetPWMSyncKill
********************************************************************************
*
* Summary:
*  Writes the register that controls whether the PWM kill signal (stop input)
*  causes asynchronous or synchronous kill operation.  By default the kill
*  operation is asynchronous.  This functionality is only applicable to the PWM
*  and PWM with dead time modes.
*
*  For Synchronous mode the kill signal disables both the line and line_n
*  signals until the next terminal count.
*
*  For Asynchronous mode the kill signal disables both the line and line_n
*  signals when the kill signal is present.  This mode should only be used
*  when the kill signal (stop input) is configured in the pass through mode
*  (Level sensitive signal).

*
* Parameters:
*  syncKillEnable
*   Values:
*     - 0 - Asynchronous
*     - 1 - Synchronous
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetPWMSyncKill(uint32 syncKillEnable)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_CONTROL_REG &= (uint32)~pGen_PWM_SYNC_KILL_MASK;
    pGen_CONTROL_REG |= ((uint32)((syncKillEnable & pGen_1BIT_MASK)  <<
                                               pGen_PWM_SYNC_KILL_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_SetPWMStopOnKill
********************************************************************************
*
* Summary:
*  Writes the register that controls whether the PWM kill signal (stop input)
*  causes the PWM counter to stop.  By default the kill operation does not stop
*  the counter.  This functionality is only applicable to the three PWM modes.
*
*
* Parameters:
*  stopOnKillEnable
*   Values:
*     - 0 - Don't stop
*     - 1 - Stop
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetPWMStopOnKill(uint32 stopOnKillEnable)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_CONTROL_REG &= (uint32)~pGen_PWM_STOP_KILL_MASK;
    pGen_CONTROL_REG |= ((uint32)((stopOnKillEnable & pGen_1BIT_MASK)  <<
                                                         pGen_PWM_STOP_KILL_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_SetPWMDeadTime
********************************************************************************
*
* Summary:
*  Writes the dead time control value.  This value delays the rising edge of
*  both the line and line_n signals the designated number of cycles resulting
*  in both signals being inactive for that many cycles.  This functionality is
*  only applicable to the PWM in the dead time mode.

*
* Parameters:
*  Dead time to insert
*   Values: 0 to 255
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetPWMDeadTime(uint32 deadTime)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_CONTROL_REG &= (uint32)~pGen_PRESCALER_MASK;
    pGen_CONTROL_REG |= ((uint32)((deadTime & pGen_8BIT_MASK) <<
                                                          pGen_PRESCALER_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_SetPWMInvert
********************************************************************************
*
* Summary:
*  Writes the bits that control whether the line and line_n outputs are
*  inverted from their normal output values.  This functionality is only
*  applicable to the three PWM modes.
*
* Parameters:
*  mask: Mask of outputs to invert.
*   Values:
*         - pGen_INVERT_LINE   - Inverts the line output
*         - pGen_INVERT_LINE_N - Inverts the line_n output
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetPWMInvert(uint32 mask)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_CONTROL_REG &= (uint32)~pGen_INV_OUT_MASK;
    pGen_CONTROL_REG |= mask;

    CyExitCriticalSection(enableInterrupts);
}



/*******************************************************************************
* Function Name: pGen_WriteCounter
********************************************************************************
*
* Summary:
*  Writes a new 16bit counter value directly into the counter register, thus
*  setting the counter (not the period) to the value written. It is not
*  advised to write to this field when the counter is running.
*
* Parameters:
*  count: value to write
*
* Return:
*  None
*
*******************************************************************************/
void pGen_WriteCounter(uint32 count)
{
    pGen_COUNTER_REG = (count & pGen_16BIT_MASK);
}


/*******************************************************************************
* Function Name: pGen_ReadCounter
********************************************************************************
*
* Summary:
*  Reads the current counter value.
*
* Parameters:
*  None
*
* Return:
*  Current counter value
*
*******************************************************************************/
uint32 pGen_ReadCounter(void)
{
    return (pGen_COUNTER_REG & pGen_16BIT_MASK);
}


/*******************************************************************************
* Function Name: pGen_SetCounterMode
********************************************************************************
*
* Summary:
*  Sets the counter mode.  Applicable to all modes except Quadrature Decoder
*  and the PWM with a pseudo random output.
*
* Parameters:
*  counterMode: Enumerated counter type values
*   Values:
*     - pGen_COUNT_UP       - Counts up
*     - pGen_COUNT_DOWN     - Counts down
*     - pGen_COUNT_UPDOWN0  - Counts up and down. Terminal count
*                                         generated when counter reaches 0
*     - pGen_COUNT_UPDOWN1  - Counts up and down. Terminal count
*                                         generated both when counter reaches 0
*                                         and period
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetCounterMode(uint32 counterMode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_CONTROL_REG &= (uint32)~pGen_UPDOWN_MASK;
    pGen_CONTROL_REG |= counterMode;

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_WritePeriod
********************************************************************************
*
* Summary:
*  Writes the 16 bit period register with the new period value.
*  To cause the counter to count for N cycles this register should be written
*  with N-1 (counts from 0 to period inclusive).
*
* Parameters:
*  period: Period value
*
* Return:
*  None
*
*******************************************************************************/
void pGen_WritePeriod(uint32 period)
{
    pGen_PERIOD_REG = (period & pGen_16BIT_MASK);
}


/*******************************************************************************
* Function Name: pGen_ReadPeriod
********************************************************************************
*
* Summary:
*  Reads the 16 bit period register.
*
* Parameters:
*  None
*
* Return:
*  Period value
*
*******************************************************************************/
uint32 pGen_ReadPeriod(void)
{
    return (pGen_PERIOD_REG & pGen_16BIT_MASK);
}


/*******************************************************************************
* Function Name: pGen_SetCompareSwap
********************************************************************************
*
* Summary:
*  Writes the register that controls whether the compare registers are
*  swapped. When enabled in the Timer/Counter mode(without capture) the swap
*  occurs at a TC event. In the PWM mode the swap occurs at the next TC event
*  following a hardware switch event.
*
* Parameters:
*  swapEnable
*   Values:
*     - 0 - Disable swap
*     - 1 - Enable swap
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetCompareSwap(uint32 swapEnable)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_CONTROL_REG &= (uint32)~pGen_RELOAD_CC_MASK;
    pGen_CONTROL_REG |= (swapEnable & pGen_1BIT_MASK);

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_WritePeriodBuf
********************************************************************************
*
* Summary:
*  Writes the 16 bit period buf register with the new period value.
*
* Parameters:
*  periodBuf: Period value
*
* Return:
*  None
*
*******************************************************************************/
void pGen_WritePeriodBuf(uint32 periodBuf)
{
    pGen_PERIOD_BUF_REG = (periodBuf & pGen_16BIT_MASK);
}


/*******************************************************************************
* Function Name: pGen_ReadPeriodBuf
********************************************************************************
*
* Summary:
*  Reads the 16 bit period buf register.
*
* Parameters:
*  None
*
* Return:
*  Period value
*
*******************************************************************************/
uint32 pGen_ReadPeriodBuf(void)
{
    return (pGen_PERIOD_BUF_REG & pGen_16BIT_MASK);
}


/*******************************************************************************
* Function Name: pGen_SetPeriodSwap
********************************************************************************
*
* Summary:
*  Writes the register that controls whether the period registers are
*  swapped. When enabled in Timer/Counter mode the swap occurs at a TC event.
*  In the PWM mode the swap occurs at the next TC event following a hardware
*  switch event.
*
* Parameters:
*  swapEnable
*   Values:
*     - 0 - Disable swap
*     - 1 - Enable swap
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetPeriodSwap(uint32 swapEnable)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_CONTROL_REG &= (uint32)~pGen_RELOAD_PERIOD_MASK;
    pGen_CONTROL_REG |= ((uint32)((swapEnable & pGen_1BIT_MASK) <<
                                                            pGen_RELOAD_PERIOD_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_WriteCompare
********************************************************************************
*
* Summary:
*  Writes the 16 bit compare register with the new compare value. Not
*  applicable for Timer/Counter with Capture or in Quadrature Decoder modes.
*
* Parameters:
*  compare: Compare value
*
* Return:
*  None
*
* Note:
*  It is not recommended to use the value equal to "0" or equal to 
*  "period value" in Center or Asymmetric align PWM modes on the 
*  PSoC 4100/PSoC 4200 devices.
*  PSoC 4000 devices write the 16 bit compare register with the decremented 
*  compare value in the Up counting mode (except 0x0u), and the incremented 
*  compare value in the Down counting mode (except 0xFFFFu).
*
*******************************************************************************/
void pGen_WriteCompare(uint32 compare)
{
    #if (pGen_CY_TCPWM_4000)
        uint32 currentMode;
    #endif /* (pGen_CY_TCPWM_4000) */

    #if (pGen_CY_TCPWM_4000)
        currentMode = ((pGen_CONTROL_REG & pGen_UPDOWN_MASK) >> pGen_UPDOWN_SHIFT);

        if (((uint32)pGen__COUNT_DOWN == currentMode) && (0xFFFFu != compare))
        {
            compare++;
        }
        else if (((uint32)pGen__COUNT_UP == currentMode) && (0u != compare))
        {
            compare--;
        }
        else
        {
        }
        
    
    #endif /* (pGen_CY_TCPWM_4000) */
    
    pGen_COMP_CAP_REG = (compare & pGen_16BIT_MASK);
}


/*******************************************************************************
* Function Name: pGen_ReadCompare
********************************************************************************
*
* Summary:
*  Reads the compare register. Not applicable for Timer/Counter with Capture
*  or in Quadrature Decoder modes.
*  PSoC 4000 devices read the incremented compare register value in the 
*  Up counting mode (except 0xFFFFu), and the decremented value in the 
*  Down counting mode (except 0x0u).
*
* Parameters:
*  None
*
* Return:
*  Compare value
*
* Note:
*  PSoC 4000 devices read the incremented compare register value in the 
*  Up counting mode (except 0xFFFFu), and the decremented value in the 
*  Down counting mode (except 0x0u).
*
*******************************************************************************/
uint32 pGen_ReadCompare(void)
{
    #if (pGen_CY_TCPWM_4000)
        uint32 currentMode;
        uint32 regVal;
    #endif /* (pGen_CY_TCPWM_4000) */

    #if (pGen_CY_TCPWM_4000)
        currentMode = ((pGen_CONTROL_REG & pGen_UPDOWN_MASK) >> pGen_UPDOWN_SHIFT);
        
        regVal = pGen_COMP_CAP_REG;
        
        if (((uint32)pGen__COUNT_DOWN == currentMode) && (0u != regVal))
        {
            regVal--;
        }
        else if (((uint32)pGen__COUNT_UP == currentMode) && (0xFFFFu != regVal))
        {
            regVal++;
        }
        else
        {
        }

        return (regVal & pGen_16BIT_MASK);
    #else
        return (pGen_COMP_CAP_REG & pGen_16BIT_MASK);
    #endif /* (pGen_CY_TCPWM_4000) */
}


/*******************************************************************************
* Function Name: pGen_WriteCompareBuf
********************************************************************************
*
* Summary:
*  Writes the 16 bit compare buffer register with the new compare value. Not
*  applicable for Timer/Counter with Capture or in Quadrature Decoder modes.
*
* Parameters:
*  compareBuf: Compare value
*
* Return:
*  None
*
* Note:
*  It is not recommended to use the value equal to "0" or equal to 
*  "period value" in Center or Asymmetric align PWM modes on the 
*  PSoC 4100/PSoC 4200 devices.
*  PSoC 4000 devices write the 16 bit compare register with the decremented 
*  compare value in the Up counting mode (except 0x0u), and the incremented 
*  compare value in the Down counting mode (except 0xFFFFu).
*
*******************************************************************************/
void pGen_WriteCompareBuf(uint32 compareBuf)
{
    #if (pGen_CY_TCPWM_4000)
        uint32 currentMode;
    #endif /* (pGen_CY_TCPWM_4000) */

    #if (pGen_CY_TCPWM_4000)
        currentMode = ((pGen_CONTROL_REG & pGen_UPDOWN_MASK) >> pGen_UPDOWN_SHIFT);

        if (((uint32)pGen__COUNT_DOWN == currentMode) && (0xFFFFu != compareBuf))
        {
            compareBuf++;
        }
        else if (((uint32)pGen__COUNT_UP == currentMode) && (0u != compareBuf))
        {
            compareBuf --;
        }
        else
        {
        }
    #endif /* (pGen_CY_TCPWM_4000) */
    
    pGen_COMP_CAP_BUF_REG = (compareBuf & pGen_16BIT_MASK);
}


/*******************************************************************************
* Function Name: pGen_ReadCompareBuf
********************************************************************************
*
* Summary:
*  Reads the compare buffer register. Not applicable for Timer/Counter with
*  Capture or in Quadrature Decoder modes.
*
* Parameters:
*  None
*
* Return:
*  Compare buffer value
*
* Note:
*  PSoC 4000 devices read the incremented compare register value in the 
*  Up counting mode (except 0xFFFFu), and the decremented value in the 
*  Down counting mode (except 0x0u).
*
*******************************************************************************/
uint32 pGen_ReadCompareBuf(void)
{
    #if (pGen_CY_TCPWM_4000)
        uint32 currentMode;
        uint32 regVal;
    #endif /* (pGen_CY_TCPWM_4000) */

    #if (pGen_CY_TCPWM_4000)
        currentMode = ((pGen_CONTROL_REG & pGen_UPDOWN_MASK) >> pGen_UPDOWN_SHIFT);

        regVal = pGen_COMP_CAP_BUF_REG;
        
        if (((uint32)pGen__COUNT_DOWN == currentMode) && (0u != regVal))
        {
            regVal--;
        }
        else if (((uint32)pGen__COUNT_UP == currentMode) && (0xFFFFu != regVal))
        {
            regVal++;
        }
        else
        {
        }

        return (regVal & pGen_16BIT_MASK);
    #else
        return (pGen_COMP_CAP_BUF_REG & pGen_16BIT_MASK);
    #endif /* (pGen_CY_TCPWM_4000) */
}


/*******************************************************************************
* Function Name: pGen_ReadCapture
********************************************************************************
*
* Summary:
*  Reads the captured counter value. This API is applicable only for
*  Timer/Counter with the capture mode and Quadrature Decoder modes.
*
* Parameters:
*  None
*
* Return:
*  Capture value
*
*******************************************************************************/
uint32 pGen_ReadCapture(void)
{
    return (pGen_COMP_CAP_REG & pGen_16BIT_MASK);
}


/*******************************************************************************
* Function Name: pGen_ReadCaptureBuf
********************************************************************************
*
* Summary:
*  Reads the capture buffer register. This API is applicable only for
*  Timer/Counter with the capture mode and Quadrature Decoder modes.
*
* Parameters:
*  None
*
* Return:
*  Capture buffer value
*
*******************************************************************************/
uint32 pGen_ReadCaptureBuf(void)
{
    return (pGen_COMP_CAP_BUF_REG & pGen_16BIT_MASK);
}


/*******************************************************************************
* Function Name: pGen_SetCaptureMode
********************************************************************************
*
* Summary:
*  Sets the capture trigger mode. For PWM mode this is the switch input.
*  This input is not applicable to the Timer/Counter without Capture and
*  Quadrature Decoder modes.
*
* Parameters:
*  triggerMode: Enumerated trigger mode value
*   Values:
*     - pGen_TRIG_LEVEL     - Level
*     - pGen_TRIG_RISING    - Rising edge
*     - pGen_TRIG_FALLING   - Falling edge
*     - pGen_TRIG_BOTH      - Both rising and falling edge
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetCaptureMode(uint32 triggerMode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_TRIG_CONTROL1_REG &= (uint32)~pGen_CAPTURE_MASK;
    pGen_TRIG_CONTROL1_REG |= triggerMode;

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_SetReloadMode
********************************************************************************
*
* Summary:
*  Sets the reload trigger mode. For Quadrature Decoder mode this is the index
*  input.
*
* Parameters:
*  triggerMode: Enumerated trigger mode value
*   Values:
*     - pGen_TRIG_LEVEL     - Level
*     - pGen_TRIG_RISING    - Rising edge
*     - pGen_TRIG_FALLING   - Falling edge
*     - pGen_TRIG_BOTH      - Both rising and falling edge
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetReloadMode(uint32 triggerMode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_TRIG_CONTROL1_REG &= (uint32)~pGen_RELOAD_MASK;
    pGen_TRIG_CONTROL1_REG |= ((uint32)(triggerMode << pGen_RELOAD_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_SetStartMode
********************************************************************************
*
* Summary:
*  Sets the start trigger mode. For Quadrature Decoder mode this is the
*  phiB input.
*
* Parameters:
*  triggerMode: Enumerated trigger mode value
*   Values:
*     - pGen_TRIG_LEVEL     - Level
*     - pGen_TRIG_RISING    - Rising edge
*     - pGen_TRIG_FALLING   - Falling edge
*     - pGen_TRIG_BOTH      - Both rising and falling edge
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetStartMode(uint32 triggerMode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_TRIG_CONTROL1_REG &= (uint32)~pGen_START_MASK;
    pGen_TRIG_CONTROL1_REG |= ((uint32)(triggerMode << pGen_START_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_SetStopMode
********************************************************************************
*
* Summary:
*  Sets the stop trigger mode. For PWM mode this is the kill input.
*
* Parameters:
*  triggerMode: Enumerated trigger mode value
*   Values:
*     - pGen_TRIG_LEVEL     - Level
*     - pGen_TRIG_RISING    - Rising edge
*     - pGen_TRIG_FALLING   - Falling edge
*     - pGen_TRIG_BOTH      - Both rising and falling edge
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetStopMode(uint32 triggerMode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_TRIG_CONTROL1_REG &= (uint32)~pGen_STOP_MASK;
    pGen_TRIG_CONTROL1_REG |= ((uint32)(triggerMode << pGen_STOP_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_SetCountMode
********************************************************************************
*
* Summary:
*  Sets the count trigger mode. For Quadrature Decoder mode this is the phiA
*  input.
*
* Parameters:
*  triggerMode: Enumerated trigger mode value
*   Values:
*     - pGen_TRIG_LEVEL     - Level
*     - pGen_TRIG_RISING    - Rising edge
*     - pGen_TRIG_FALLING   - Falling edge
*     - pGen_TRIG_BOTH      - Both rising and falling edge
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetCountMode(uint32 triggerMode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_TRIG_CONTROL1_REG &= (uint32)~pGen_COUNT_MASK;
    pGen_TRIG_CONTROL1_REG |= ((uint32)(triggerMode << pGen_COUNT_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_TriggerCommand
********************************************************************************
*
* Summary:
*  Triggers the designated command to occur on the designated TCPWM instances.
*  The mask can be used to apply this command simultaneously to more than one
*  instance.  This allows multiple TCPWM instances to be synchronized.
*
* Parameters:
*  mask: A combination of mask bits for each instance of the TCPWM that the
*        command should apply to.  This function from one instance can be used
*        to apply the command to any of the instances in the design.
*        The mask value for a specific instance is available with the MASK
*        define.
*  command: Enumerated command values. Capture command only applicable for
*           Timer/Counter with Capture and PWM modes.
*   Values:
*     - pGen_CMD_CAPTURE    - Trigger Capture/Switch command
*     - pGen_CMD_RELOAD     - Trigger Reload/Index command
*     - pGen_CMD_STOP       - Trigger Stop/Kill command
*     - pGen_CMD_START      - Trigger Start/phiB command
*
* Return:
*  None
*
*******************************************************************************/
void pGen_TriggerCommand(uint32 mask, uint32 command)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    pGen_COMMAND_REG = ((uint32)(mask << command));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: pGen_ReadStatus
********************************************************************************
*
* Summary:
*  Reads the status of the pGen.
*
* Parameters:
*  None
*
* Return:
*  Status
*   Values:
*     - pGen_STATUS_DOWN    - Set if counting down
*     - pGen_STATUS_RUNNING - Set if counter is running
*
*******************************************************************************/
uint32 pGen_ReadStatus(void)
{
    return ((pGen_STATUS_REG >> pGen_RUNNING_STATUS_SHIFT) |
            (pGen_STATUS_REG & pGen_STATUS_DOWN));
}


/*******************************************************************************
* Function Name: pGen_SetInterruptMode
********************************************************************************
*
* Summary:
*  Sets the interrupt mask to control which interrupt
*  requests generate the interrupt signal.
*
* Parameters:
*   interruptMask: Mask of bits to be enabled
*   Values:
*     - pGen_INTR_MASK_TC       - Terminal count mask
*     - pGen_INTR_MASK_CC_MATCH - Compare count / capture mask
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetInterruptMode(uint32 interruptMask)
{
    pGen_INTERRUPT_MASK_REG =  interruptMask;
}


/*******************************************************************************
* Function Name: pGen_GetInterruptSourceMasked
********************************************************************************
*
* Summary:
*  Gets the interrupt requests masked by the interrupt mask.
*
* Parameters:
*   None
*
* Return:
*  Masked interrupt source
*   Values:
*     - pGen_INTR_MASK_TC       - Terminal count mask
*     - pGen_INTR_MASK_CC_MATCH - Compare count / capture mask
*
*******************************************************************************/
uint32 pGen_GetInterruptSourceMasked(void)
{
    return (pGen_INTERRUPT_MASKED_REG);
}


/*******************************************************************************
* Function Name: pGen_GetInterruptSource
********************************************************************************
*
* Summary:
*  Gets the interrupt requests (without masking).
*
* Parameters:
*  None
*
* Return:
*  Interrupt request value
*   Values:
*     - pGen_INTR_MASK_TC       - Terminal count mask
*     - pGen_INTR_MASK_CC_MATCH - Compare count / capture mask
*
*******************************************************************************/
uint32 pGen_GetInterruptSource(void)
{
    return (pGen_INTERRUPT_REQ_REG);
}


/*******************************************************************************
* Function Name: pGen_ClearInterrupt
********************************************************************************
*
* Summary:
*  Clears the interrupt request.
*
* Parameters:
*   interruptMask: Mask of interrupts to clear
*   Values:
*     - pGen_INTR_MASK_TC       - Terminal count mask
*     - pGen_INTR_MASK_CC_MATCH - Compare count / capture mask
*
* Return:
*  None
*
*******************************************************************************/
void pGen_ClearInterrupt(uint32 interruptMask)
{
    pGen_INTERRUPT_REQ_REG = interruptMask;
}


/*******************************************************************************
* Function Name: pGen_SetInterrupt
********************************************************************************
*
* Summary:
*  Sets a software interrupt request.
*
* Parameters:
*   interruptMask: Mask of interrupts to set
*   Values:
*     - pGen_INTR_MASK_TC       - Terminal count mask
*     - pGen_INTR_MASK_CC_MATCH - Compare count / capture mask
*
* Return:
*  None
*
*******************************************************************************/
void pGen_SetInterrupt(uint32 interruptMask)
{
    pGen_INTERRUPT_SET_REG = interruptMask;
}


/* [] END OF FILE */
