/*******************************************************************************
* File Name: load_res.c  
* Version 2.20
*
* Description:
*  This file contains APIs to set up the Pins component for low power modes.
*
* Note:
*
********************************************************************************
* Copyright 2015, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions, 
* disclaimers, and limitations in the end user license agreement accompanying 
* the software package with which this file was provided.
*******************************************************************************/

#include "cytypes.h"
#include "load_res.h"

static load_res_BACKUP_STRUCT  load_res_backup = {0u, 0u, 0u};


/*******************************************************************************
* Function Name: load_res_Sleep
****************************************************************************//**
*
* \brief Stores the pin configuration and prepares the pin for entering chip 
*  deep-sleep/hibernate modes. This function applies only to SIO and USBIO pins.
*  It should not be called for GPIO or GPIO_OVT pins.
*
* <b>Note</b> This function is available in PSoC 4 only.
*
* \return 
*  None 
*  
* \sideeffect
*  For SIO pins, this function configures the pin input threshold to CMOS and
*  drive level to Vddio. This is needed for SIO pins when in device 
*  deep-sleep/hibernate modes.
*
* \funcusage
*  \snippet load_res_SUT.c usage_load_res_Sleep_Wakeup
*******************************************************************************/
void load_res_Sleep(void)
{
    #if defined(load_res__PC)
        load_res_backup.pcState = load_res_PC;
    #else
        #if (CY_PSOC4_4200L)
            /* Save the regulator state and put the PHY into suspend mode */
            load_res_backup.usbState = load_res_CR1_REG;
            load_res_USB_POWER_REG |= load_res_USBIO_ENTER_SLEEP;
            load_res_CR1_REG &= load_res_USBIO_CR1_OFF;
        #endif
    #endif
    #if defined(CYIPBLOCK_m0s8ioss_VERSION) && defined(load_res__SIO)
        load_res_backup.sioState = load_res_SIO_REG;
        /* SIO requires unregulated output buffer and single ended input buffer */
        load_res_SIO_REG &= (uint32)(~load_res_SIO_LPM_MASK);
    #endif  
}


/*******************************************************************************
* Function Name: load_res_Wakeup
****************************************************************************//**
*
* \brief Restores the pin configuration that was saved during Pin_Sleep(). This 
* function applies only to SIO and USBIO pins. It should not be called for
* GPIO or GPIO_OVT pins.
*
* For USBIO pins, the wakeup is only triggered for falling edge interrupts.
*
* <b>Note</b> This function is available in PSoC 4 only.
*
* \return 
*  None
*  
* \funcusage
*  Refer to load_res_Sleep() for an example usage.
*******************************************************************************/
void load_res_Wakeup(void)
{
    #if defined(load_res__PC)
        load_res_PC = load_res_backup.pcState;
    #else
        #if (CY_PSOC4_4200L)
            /* Restore the regulator state and come out of suspend mode */
            load_res_USB_POWER_REG &= load_res_USBIO_EXIT_SLEEP_PH1;
            load_res_CR1_REG = load_res_backup.usbState;
            load_res_USB_POWER_REG &= load_res_USBIO_EXIT_SLEEP_PH2;
        #endif
    #endif
    #if defined(CYIPBLOCK_m0s8ioss_VERSION) && defined(load_res__SIO)
        load_res_SIO_REG = load_res_backup.sioState;
    #endif
}


/* [] END OF FILE */
