/*******************************************************************************
* File Name: ext_trig.c  
* Version 2.20
*
* Description:
*  This file contains APIs to set up the Pins component for low power modes.
*
* Note:
*
********************************************************************************
* Copyright 2015, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions, 
* disclaimers, and limitations in the end user license agreement accompanying 
* the software package with which this file was provided.
*******************************************************************************/

#include "cytypes.h"
#include "ext_trig.h"

static ext_trig_BACKUP_STRUCT  ext_trig_backup = {0u, 0u, 0u};


/*******************************************************************************
* Function Name: ext_trig_Sleep
****************************************************************************//**
*
* \brief Stores the pin configuration and prepares the pin for entering chip 
*  deep-sleep/hibernate modes. This function applies only to SIO and USBIO pins.
*  It should not be called for GPIO or GPIO_OVT pins.
*
* <b>Note</b> This function is available in PSoC 4 only.
*
* \return 
*  None 
*  
* \sideeffect
*  For SIO pins, this function configures the pin input threshold to CMOS and
*  drive level to Vddio. This is needed for SIO pins when in device 
*  deep-sleep/hibernate modes.
*
* \funcusage
*  \snippet ext_trig_SUT.c usage_ext_trig_Sleep_Wakeup
*******************************************************************************/
void ext_trig_Sleep(void)
{
    #if defined(ext_trig__PC)
        ext_trig_backup.pcState = ext_trig_PC;
    #else
        #if (CY_PSOC4_4200L)
            /* Save the regulator state and put the PHY into suspend mode */
            ext_trig_backup.usbState = ext_trig_CR1_REG;
            ext_trig_USB_POWER_REG |= ext_trig_USBIO_ENTER_SLEEP;
            ext_trig_CR1_REG &= ext_trig_USBIO_CR1_OFF;
        #endif
    #endif
    #if defined(CYIPBLOCK_m0s8ioss_VERSION) && defined(ext_trig__SIO)
        ext_trig_backup.sioState = ext_trig_SIO_REG;
        /* SIO requires unregulated output buffer and single ended input buffer */
        ext_trig_SIO_REG &= (uint32)(~ext_trig_SIO_LPM_MASK);
    #endif  
}


/*******************************************************************************
* Function Name: ext_trig_Wakeup
****************************************************************************//**
*
* \brief Restores the pin configuration that was saved during Pin_Sleep(). This 
* function applies only to SIO and USBIO pins. It should not be called for
* GPIO or GPIO_OVT pins.
*
* For USBIO pins, the wakeup is only triggered for falling edge interrupts.
*
* <b>Note</b> This function is available in PSoC 4 only.
*
* \return 
*  None
*  
* \funcusage
*  Refer to ext_trig_Sleep() for an example usage.
*******************************************************************************/
void ext_trig_Wakeup(void)
{
    #if defined(ext_trig__PC)
        ext_trig_PC = ext_trig_backup.pcState;
    #else
        #if (CY_PSOC4_4200L)
            /* Restore the regulator state and come out of suspend mode */
            ext_trig_USB_POWER_REG &= ext_trig_USBIO_EXIT_SLEEP_PH1;
            ext_trig_CR1_REG = ext_trig_backup.usbState;
            ext_trig_USB_POWER_REG &= ext_trig_USBIO_EXIT_SLEEP_PH2;
        #endif
    #endif
    #if defined(CYIPBLOCK_m0s8ioss_VERSION) && defined(ext_trig__SIO)
        ext_trig_SIO_REG = ext_trig_backup.sioState;
    #endif
}


/* [] END OF FILE */
