/*******************************************************************************
* File Name: IO_low.c  
* Version 2.20
*
* Description:
*  This file contains APIs to set up the Pins component for low power modes.
*
* Note:
*
********************************************************************************
* Copyright 2015, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions, 
* disclaimers, and limitations in the end user license agreement accompanying 
* the software package with which this file was provided.
*******************************************************************************/

#include "cytypes.h"
#include "IO_low.h"

static IO_low_BACKUP_STRUCT  IO_low_backup = {0u, 0u, 0u};


/*******************************************************************************
* Function Name: IO_low_Sleep
****************************************************************************//**
*
* \brief Stores the pin configuration and prepares the pin for entering chip 
*  deep-sleep/hibernate modes. This function applies only to SIO and USBIO pins.
*  It should not be called for GPIO or GPIO_OVT pins.
*
* <b>Note</b> This function is available in PSoC 4 only.
*
* \return 
*  None 
*  
* \sideeffect
*  For SIO pins, this function configures the pin input threshold to CMOS and
*  drive level to Vddio. This is needed for SIO pins when in device 
*  deep-sleep/hibernate modes.
*
* \funcusage
*  \snippet IO_low_SUT.c usage_IO_low_Sleep_Wakeup
*******************************************************************************/
void IO_low_Sleep(void)
{
    #if defined(IO_low__PC)
        IO_low_backup.pcState = IO_low_PC;
    #else
        #if (CY_PSOC4_4200L)
            /* Save the regulator state and put the PHY into suspend mode */
            IO_low_backup.usbState = IO_low_CR1_REG;
            IO_low_USB_POWER_REG |= IO_low_USBIO_ENTER_SLEEP;
            IO_low_CR1_REG &= IO_low_USBIO_CR1_OFF;
        #endif
    #endif
    #if defined(CYIPBLOCK_m0s8ioss_VERSION) && defined(IO_low__SIO)
        IO_low_backup.sioState = IO_low_SIO_REG;
        /* SIO requires unregulated output buffer and single ended input buffer */
        IO_low_SIO_REG &= (uint32)(~IO_low_SIO_LPM_MASK);
    #endif  
}


/*******************************************************************************
* Function Name: IO_low_Wakeup
****************************************************************************//**
*
* \brief Restores the pin configuration that was saved during Pin_Sleep(). This 
* function applies only to SIO and USBIO pins. It should not be called for
* GPIO or GPIO_OVT pins.
*
* For USBIO pins, the wakeup is only triggered for falling edge interrupts.
*
* <b>Note</b> This function is available in PSoC 4 only.
*
* \return 
*  None
*  
* \funcusage
*  Refer to IO_low_Sleep() for an example usage.
*******************************************************************************/
void IO_low_Wakeup(void)
{
    #if defined(IO_low__PC)
        IO_low_PC = IO_low_backup.pcState;
    #else
        #if (CY_PSOC4_4200L)
            /* Restore the regulator state and come out of suspend mode */
            IO_low_USB_POWER_REG &= IO_low_USBIO_EXIT_SLEEP_PH1;
            IO_low_CR1_REG = IO_low_backup.usbState;
            IO_low_USB_POWER_REG &= IO_low_USBIO_EXIT_SLEEP_PH2;
        #endif
    #endif
    #if defined(CYIPBLOCK_m0s8ioss_VERSION) && defined(IO_low__SIO)
        IO_low_SIO_REG = IO_low_backup.sioState;
    #endif
}


/* [] END OF FILE */
